<?php
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class Wawp_Abandoned_Cart_Notifications {

    private $option_name = 'wawp_abandoned_cart_notification_settings';

    public function __construct() {
        add_action( 'wawp_send_abandoned_cart_notification_step', array( $this, 'send_notification_step' ), 10, 3 );
        add_action( 'admin_init', array( $this, 'handle_save_settings' ) );
    }

    public function handle_save_settings() {
    if ( ! current_user_can( 'manage_options' ) ) {
        error_log('AWP AC Save: Permission check failed.');  
        return;
    }

    if ( isset( $_POST['wawp_ac_nonce'] ) ) {
        error_log('AWP AC Save: Nonce found.');  
        if ( ! wp_verify_nonce( $_POST['wawp_ac_nonce'], 'wawp_ac_settings_action' ) ) {
            error_log('AWP AC Save: Nonce verification FAILED.');  
            wp_die( 'Invalid nonce specified', 'Error', ['response' => 403] );
        }
        error_log('AWP AC Save: Nonce verified.');  

        $input = $_POST[$this->option_name] ?? [];
        error_log('AWP AC Save: Raw input received: ' . print_r($input, true)); 
        $input = wp_unslash($input);
        error_log('AWP AC Save: Unslashed input: ' . print_r($input, true)); 
        
        if (empty($input) && $_SERVER['CONTENT_LENGTH'] > 1000) {
             error_log('AWP AC Save: WARNING - Input array is empty but POST Content-Length was large. Possible early truncation (max_input_vars?).');
        }

        $sanitized_input = $this->sanitize_settings($input);
        error_log('AWP AC Save: Sanitized input: ' . print_r($sanitized_input, true));  

        error_log('AWP AC Save: Attempting update_option...');  
        $update_result = update_option($this->option_name, $sanitized_input);
        error_log('AWP AC Save: update_option result: ' . ($update_result ? 'Success (or no change)' : 'Failed'));  

        $check_option = get_option($this->option_name);
        error_log('AWP AC Save: Option value after update attempt: ' . print_r($check_option, true));  


        add_action('admin_notices', function() {
           echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__( 'Settings saved successfully!', 'awp' ) . '</p></div>';
        });
    } else {
        error_log('AWP AC Save: Nonce field not found.');  
    }
    }


    public function settings_page() {

        $banned_msg  = get_transient('siteB_banned_msg');
        $token       = get_option('mysso_token');
        $user_data   = get_transient('siteB_user_data');
        if ($banned_msg) {
            echo '<div class="wrap"><h1><i class="ri-lock-line"></i> ' . esc_html__('Whatsapp Activity Logs', 'awp') . '</h1><p style="color:red;">' . esc_html(Wawp_Global_Messages::get('blocked_generic')) . '</p></div>';
            return;
        }
        if (!$token) {
            echo '<div class="wrap"><h1><i class="dashicons dashicons-lock"></i> ' . esc_html__('Whatsapp Activity Logs', 'awp') . '</h1><p>' . esc_html(Wawp_Global_Messages::get('need_login')) . '</p></div>';
            return;
        }
        $current_domain = parse_url(get_site_url(), PHP_URL_HOST);
        if ($user_data && isset($user_data['sites'][$current_domain]) && $user_data['sites'][$current_domain] !== 'active') {
            echo '<div class="wrap"><h1><i class="ri-lock-line"></i> ' . esc_html__('Whatsapp Activity Logs', 'awp') . '</h1><p style="color:red;">' . esc_html(Wawp_Global_Messages::get('not_active_site')) . '</p></div>';
            return;
        }
        if ( ! AWP_Admin_Notices::require_active_sender('Abandoned Carts Reminders') ) {return; }
        if ( ! AWP_Admin_Notices::require_online_instance( null) ) {return;}
        
        
        
        ?>
        <div class="wrap">
            
                <div class="page-header_row">
                    <div class="page-header">
                        <h2 class="page-title"><?php _e( 'Abandoned Cart Reminders', 'awp' ); ?></h2>
                        <p class="page-title"><?php _e( 'Create multi-step notification sequences to recover potential customers and abandoned carts.', 'awp' ); ?></p>
                    </div>
            </div>
            
            
            <form id="wawp-ac-form" method="post" action="">
                <?php wp_nonce_field('wawp_ac_settings_action', 'wawp_ac_nonce'); ?>

                <div class="wawp-setting-card coupon_set">
                     <div class="wawp-setting-header awp-toggle-switch">
                        <div class="card-header">
                            <div class="card-title"><i class="ri-coupon-3-line"></i> <?php _e( 'Coupon Code Settings', 'awp' ); ?></div>
                        </div>
                        <div class="btn-group"><i class="ri-arrow-down-s-line toggled-arrow"></i></div>
                    </div>
                    <div class="wawp-setting-content" style="display: none;">
                        <table class="form-table">
                            <tbody>
                                <tr>
                                    <th scope="row"><?php _e( 'Enable Coupon Generation', 'awp' ); ?></th>
                                    <td><div class="btn-group"><?php $this->render_coupon_enable_field(); ?></div></td>
                                </tr>
                                <tr>
                                    <th scope="row"><?php _e( 'Coupon Value', 'awp' ); ?></th>
                                    <td><div class="btn-group"><?php $this->render_coupon_value_field(); ?></div></td>
                                </tr>
                                <tr>
                                    <th scope="row"><?php _e( 'Coupon Expires After', 'awp' ); ?></th>
                                    <td><div class="btn-group"><?php $this->render_coupon_expiry_field(); ?></div></td>
                                </tr>
                                <tr>
                                    <th scope="row"><?php _e( 'Allow free shipping', 'awp' ); ?></th>
                                    <td><div class="btn-group"><?php $this->render_coupon_free_shipping_field(); ?></div></td>
                                </tr>
                                <tr>
                                    <th scope="row"><?php _e( 'Individual use only', 'awp' ); ?></th>
                                    <td><div class="btn-group"><?php $this->render_coupon_individual_use_field(); ?></div></td>
                                </tr>
                                <tr>
                                    <th scope="row"><?php _e( 'Minimum spend', 'awp' ); ?></th>
                                    <td><div class="btn-group"><?php $this->render_coupon_minimum_spend_field(); ?></div></td>
                                </tr>
                                <tr>
                                    <th scope="row"><?php _e( 'Exclude sale items', 'awp' ); ?></th>
                                    <td><div class="btn-group"><?php $this->render_coupon_exclude_sale_items_field(); ?></div></td>
                                </tr>
                            </tbody>
                        </table>
                    </div>
                </div>

                <?php
                $this->render_notification_sequence_ui(['type' => 'potential_customer', 'title' => __( '[ Potential Customer ] Status Notifications', 'awp' )]);
                $this->render_notification_sequence_ui(['type' => 'abandoned_cart', 'title' => __( 'Abandoned Cart Reminders for [ Shopping ] Status', 'awp' )]);
                submit_button();
                ?>
            </form>
        </div>
        <?php
    }

    public function render_notification_sequence_ui($args) {
        $type = $args['type'];
        $title = $args['title'];
        $options = get_option($this->option_name, []);
        $notifications = $options[$type]['notifications'] ?? [];

        ?>
        <div class="wawp-setting-card">
            <div class="wawp-setting-header awp-toggle-switch">
                <div class="card-header">
                    <div class="card-title"><i class="ri-list-check-2"></i> <?php echo esc_html($title); ?></div>
                </div>
                <div class="btn-group"><i class="ri-arrow-down-s-line toggled-arrow"></i></div>
            </div>
            <div class="wawp-setting-content" style="display: none;">
                <div id="wawp-ac-<?php echo esc_attr($type); ?>-wrapper" class="notification-sequence-wrapper">
                    <?php
                    if (empty($notifications)) {
                        echo $this->get_notification_card_html($type, 0, []);
                    } else {
                        foreach ($notifications as $index => $notification_data) {
                            echo $this->get_notification_card_html($type, $index, $notification_data);
                        }
                    }
                    ?>
                </div>
                <button type="button" class="awp-btn secondary add-notification-step" style="height: 36px;" data-type="<?php echo esc_attr($type); ?>"><i class="ri-add-line"></i> <?php _e('Add Notification Step', 'awp'); ?></button>
            </div>
        </div>
        <?php
    }

    private function get_notification_card_html($type, $index, $data) {
        ob_start();
        $options = get_option($this->option_name, []);
        $coupon_enabled = isset($options['coupon_settings']['enable']) && $options['coupon_settings']['enable'] == 1;
        
        $db_manager = new AWP_Database_Manager();
            $senders_opts = get_option('awp_senders_enabled', ['email' => 1, 'wa' => 1, 'meta' => 1]);
            $is_email_sender_enabled = !empty($senders_opts['email']);
            $is_wa_sender_enabled    = !empty($senders_opts['wa']);
            $is_meta_sender_enabled  = !empty($senders_opts['meta']);
        
        $is_wawp_active = $this->is_wawp_active();
        $is_meta_sender_active = isset($db_manager->tables['mws_settings']);
        
        $selected_channels = $data['send_via'] ?? ($is_wawp_active ? ['wawp_whatsapp'] : []);
        if (!is_array($selected_channels)) {
            $selected_channels = [$selected_channels];
        }

        ?>
        <div class="notification-card postbox closed">
            <div class="postbox-header">
                <h2 class="hndle">
                    <span><?php printf(__('Step %d', 'awp'), $index + 1); ?></span>
                </h2>
                <div class="handle-actions">
                    <label class="awp-switch">
                        <input type="hidden" name="<?php echo esc_attr($this->option_name); ?>[<?php echo esc_attr($type); ?>][notifications][<?php echo $index; ?>][enable]" value="0" />
                        <input type="checkbox" class="wawp-rule-enable-switch" name="<?php echo esc_attr($this->option_name); ?>[<?php echo esc_attr($type); ?>][notifications][<?php echo $index; ?>][enable]" value="1" <?php checked(1, $data['enable'] ?? 1); ?> />
                        <span class="awp-slider"></span>
                    </label>
                    <button type="button" class="handlediv" aria-expanded="false"><span class="screen-reader-text"><?php _e('Toggle panel', 'awp');?></span><span class="toggle-indicator" aria-hidden="true"></span></button>
                </div>
            </div>
            <div class="inside">
                <div class="wawp-notif-card-content">
                    <?php if ($coupon_enabled): ?>
                    <table class="form-table">
                        <tr valign="top">
                            <th scope="row"><label><i class="ri-coupon-3-line"></i> <?php _e('Attach Coupon Code', 'awp'); ?></label></th>
                            <td>
                                <div class="btn-group">
                                    <label class="awp-switch">
                                        <input type="hidden" name="<?php echo esc_attr($this->option_name); ?>[<?php echo esc_attr($type); ?>][notifications][<?php echo $index; ?>][use_coupon]" value="0" />
                                        <input type="checkbox" name="<?php echo esc_attr($this->option_name); ?>[<?php echo esc_attr($type); ?>][notifications][<?php echo $index; ?>][use_coupon]" value="1" <?php checked(1, $data['use_coupon'] ?? 0); ?> />
                                        <span class="awp-slider"></span>
                                    </label>
                                    <p class="description" style="display: inline-block; margin-left: 10px;"><?php _e('If enabled, a unique coupon code will be generated.', 'awp'); ?></p>
                                </div>
                            </td>
                        </tr>
                    </table>
                    <?php endif; ?>
                    <table class="form-table">
                        <tr valign="top">
                            <th scope="row">
                                <label><i class="ri-time-line"></i> <?php _e('Waiting Time', 'awp'); ?></label>
                            </th>
                            <td>
                                <select name="<?php echo esc_attr($this->option_name); ?>[<?php echo esc_attr($type); ?>][notifications][<?php echo $index; ?>][send_timing]" class="wawp-notif-send-timing-select">
                                    <option value="instant" <?php selected($data['send_timing'] ?? 'instant', 'instant'); ?>><?php _e('Instant', 'awp'); ?></option>
                                    <option value="delayed" <?php selected($data['send_timing'] ?? 'instant', 'delayed'); ?>><?php _e('After a Delay', 'awp'); ?></option>
                                </select>
                                <div class="delay-fields" style="<?php echo ($data['send_timing'] ?? 'instant') === 'delayed' ? 'display:flex;' : 'display:none;'; ?>">
                                    <input type="number" name="<?php echo esc_attr($this->option_name); ?>[<?php echo esc_attr($type); ?>][notifications][<?php echo $index; ?>][delay_value]" value="<?php echo esc_attr($data['delay_value'] ?? '1'); ?>" min="1" style="width:70px;" />
                                    <select name="<?php echo esc_attr($this->option_name); ?>[<?php echo esc_attr($type); ?>][notifications][<?php echo $index; ?>][delay_unit]">
                                        <option value="minutes" <?php selected($data['delay_unit'] ?? 'minutes', 'minutes'); ?>><?php _e('Minutes', 'awp'); ?></option>
                                        <option value="hours" <?php selected($data['delay_unit'] ?? '', 'hours'); ?>><?php _e('Hours', 'awp'); ?></option>
                                        <option value="days" <?php selected($data['delay_unit'] ?? '', 'days'); ?>><?php _e('Days', 'awp'); ?></option>
                                    </select>
                                </div>
                            </td>
                        </tr>
                        <tr valign="top">
                            <th scope="row">
                                <label><i class="ri-send-plane-line"></i> <?php _e('Send Via', 'awp'); ?></label>
                            </th>
                            <td>
                                <select name="<?php echo esc_attr($this->option_name); ?>[<?php echo esc_attr($type); ?>][notifications][<?php echo $index; ?>][send_via][]" class="wac-send-via-select" multiple="multiple" style="width:100%;">
                              <?php if ($is_wa_sender_enabled): ?>
                                    <option value="wawp_whatsapp" <?php selected(in_array('wawp_whatsapp', $selected_channels)); ?>><?php _e('Wawp WhatsApp', 'awp'); ?></option>
                                    <?php endif; ?>
                                    <?php if ($is_meta_sender_enabled): ?>
                                    <option value="meta_whatsapp" <?php selected(in_array('meta_whatsapp', $selected_channels)); ?>><?php _e('Meta WhatsApp', 'awp'); ?></option>
                                    <?php endif; ?>
                                    <?php if ($is_email_sender_enabled): ?>
                                    <option value="email" <?php selected(in_array('email', $selected_channels)); ?>><?php _e('Email', 'awp'); ?></option>
                                    <?php endif; ?>
                                </select>
                            </td>
                        </tr>
                    </table>

                    <div class="channel-fields-wrapper">
                        <!-- Wawp WhatsApp Fields -->
                        <div class="wawp-setting-card channel-fields" data-channel="wawp_whatsapp" style="<?php echo in_array('wawp_whatsapp', $selected_channels) ? '' : 'display:none;'; ?>">
                            <div class="wawp-setting-header awp-toggle-switch">
                                <div class="card-header">
                                    <div class="card-title"><i class="ri-whatsapp-line"></i> <?php _e('Wawp WhatsApp Message', 'awp'); ?></div>
                                </div>
                                <div class="btn-group"><i class="ri-arrow-down-s-line toggled-arrow"></i></div>
                            </div>
                            <div class="wawp-setting-content" style="display: none;">
                                <textarea name="<?php echo esc_attr($this->option_name); ?>[<?php echo esc_attr($type); ?>][notifications][<?php echo $index; ?>][whatsapp_message]" rows="5" class="large-text wawp-emojione-editor"><?php echo esc_textarea($data['whatsapp_message'] ?? ''); ?></textarea>
                                <div class="placeholder-buttons"></div>
                            </div>
                        </div>
                        <!-- Meta WhatsApp Fields -->
                        <div class="wawp-setting-card channel-fields" data-channel="meta_whatsapp" style="<?php echo in_array('meta_whatsapp', $selected_channels) ? '' : 'display:none;'; ?>">
                             <div class="wawp-setting-header awp-toggle-switch">
                                <div class="card-header">
                                    <div class="card-title"><i class="ri-meta-line"></i> <?php _e('Meta WhatsApp Template', 'awp'); ?></div>
                                </div>
                                <div class="btn-group"><i class="ri-arrow-down-s-line toggled-arrow"></i></div>
                            </div>
                            <div class="wawp-setting-content" style="display: none;">
                                 <select name="<?php echo esc_attr($this->option_name); ?>[<?php echo esc_attr($type); ?>][notifications][<?php echo $index; ?>][meta_template]">
                                    <option value=""><?php _e('Select a template', 'awp'); ?></option>
                                    <?php
                                    global $wpdb;
                                    $templates_table = $db_manager->tables['mws_templates'];
                                    $templates = $wpdb->get_results("SELECT template_name, language_code FROM " . $templates_table . " ORDER BY template_name ASC, language_code ASC");
                                    if (!empty($templates)) {
                                        foreach ($templates as $template) {
                                            $template_val = $template->template_name . '|' . $template->language_code;
                                            echo '<option value="' . esc_attr($template_val) . '" ' . selected($data['meta_template'] ?? '', $template_val, false) . '>' . esc_html($template->template_name . ' (' . $template->language_code . ')') . '</option>';
                                        }
                                    }
                                    ?>
                                 </select>
                            </div>
                        </div>
                        <!-- Email Fields -->
                        <div class="wawp-setting-card channel-fields" data-channel="email" style="<?php echo in_array('email', $selected_channels) ? '' : 'display:none;'; ?>">
                            <div class="wawp-setting-header awp-toggle-switch">
                                <div class="card-header">
                                    <div class="card-title"><i class="ri-mail-line"></i> <?php _e('Email Message', 'awp'); ?></div>
                                </div>
                                <div class="btn-group"><i class="ri-arrow-down-s-line toggled-arrow"></i></div>
                            </div>
                            <div class="wawp-setting-content" style="display: none;">
                                <p><label for="<?php echo esc_attr($this->option_name) . '_' . $type . '_' . $index . '_email_subject'; ?>"><?php _e('Subject', 'awp'); ?></label></p>
                                <input type="text" id="<?php echo esc_attr($this->option_name) . '_' . $type . '_' . $index . '_email_subject'; ?>" name="<?php echo esc_attr($this->option_name); ?>[<?php echo esc_attr($type); ?>][notifications][<?php echo $index; ?>][email_subject]" value="<?php echo esc_attr($data['email_subject'] ?? ''); ?>" class="regular-text" />
                                <p style="margin-top: 1em;"><label for="<?php echo 'wawp_' . esc_attr($type) . '_' . $index . '_email_body'; ?>"><?php _e('Body', 'awp'); ?></label></p>
                                <?php
                                $editor_id = 'wawp_' . esc_attr($type) . '_' . $index . '_email_body';
                                $content = $data['email_body'] ?? '';
                                wp_editor($content, $editor_id, [
                                    'textarea_name' => esc_attr($this->option_name) . '[' . esc_attr($type) . '][notifications][' . $index . '][email_body]',
                                    'textarea_rows' => 10,
                                    'teeny' => true
                                ]);
                                ?>
                                <div class="placeholder-buttons"></div>
                            </div>
                        </div>
                        <button type="button" class="awp-btn delete-plain button-link-delete remove-notification-step" style="align-self: end;"><?php _e('Remove Step', 'awp'); ?></button>
                    </div>
                </div>
            </div>
        </div>
        <?php
        return ob_get_clean();
    }
    
public function render_coupon_enable_field() {
    $options = get_option($this->option_name, []);
    $value = (int)($options['coupon_settings']['enable'] ?? 0); 

    ?>
    <label class="awp-switch">
        <input type="hidden" name="<?php echo $this->option_name; ?>[coupon_settings][enable]" value="0" />
        <input type="checkbox"
               name="<?php echo $this->option_name; ?>[coupon_settings][enable]"
               value="1"
               <?php checked(1, $value); ?> />
        <span class="awp-slider"></span>
    </label>
    <p class="description" style="display:inline-block;margin-left:10px;">
        <?php _e('Enable this to generate unique coupon codes for notifications.', 'awp'); ?>
    </p>
    <?php
}


    public function render_coupon_value_field() {
        $options = get_option($this->option_name);
        $amount = $options['coupon_settings']['amount'] ?? '10';
        $type = $options['coupon_settings']['discount_type'] ?? 'percent';

        echo '<input type="number" name="' . $this->option_name . '[coupon_settings][amount]" value="' . esc_attr($amount) . '" min="0" step="any" style="width: 100px;" />';
        echo '<select name="' . $this->option_name . '[coupon_settings][discount_type]">';
        echo '<option value="percent" ' . selected($type, 'percent', false) . '>' . __('Percentage (%)', 'awp') . '</option>';
        echo '<option value="fixed_cart" ' . selected($type, 'fixed_cart', false) . '>' . sprintf(__('Fixed Amount (%s)', 'awp'), get_woocommerce_currency_symbol()) . '</option>';
        echo '</select>';
    }

    public function render_coupon_expiry_field() {
        $options = get_option($this->option_name);
        $value = $options['coupon_settings']['expiry_value'] ?? '7';
        $unit = $options['coupon_settings']['expiry_unit'] ?? 'days';
        
        echo '<input type="number" name="' . $this->option_name . '[coupon_settings][expiry_value]" value="' . esc_attr($value) . '" min="1" style="width: 80px;" />';
        echo '<select name="' . $this->option_name . '[coupon_settings][expiry_unit]">';
        $units = ['hours' => __('Hours', 'awp'), 'days' => __('Days', 'awp'), 'weeks' => __('Weeks', 'awp'), 'months' => __('Months', 'awp'), 'years' => __('Years', 'awp')];
        foreach ($units as $unit_key => $unit_label) {
            echo '<option value="' . esc_attr($unit_key) . '" ' . selected($unit, $unit_key, false) . '>' . esc_html($unit_label) . '</option>';
        }
        echo '</select>';
    }

public function render_coupon_free_shipping_field() {
    $options = get_option($this->option_name, []);
    $value = (int)($options['coupon_settings']['free_shipping'] ?? 0);
    ?>
    <label class="awp-switch">
        <input type="hidden" name="<?php echo $this->option_name; ?>[coupon_settings][free_shipping]" value="0" />
        <input type="checkbox"
               name="<?php echo $this->option_name; ?>[coupon_settings][free_shipping]"
               value="1"
               <?php checked(1, $value); ?> />
        <span class="awp-slider"></span>
    </label>
    <p class="description" style="display:inline-block;margin-left:10px;"><?php _e('Check this box if the coupon grants free shipping.', 'awp'); ?></p>
    <?php
}

public function render_coupon_individual_use_field() {
    $options = get_option($this->option_name, []);
    $value = (int)($options['coupon_settings']['individual_use'] ?? 0);
    ?>
    <label class="awp-switch">
        <input type="hidden" name="<?php echo $this->option_name; ?>[coupon_settings][individual_use]" value="0" />
        <input type="checkbox"
               name="<?php echo $this->option_name; ?>[coupon_settings][individual_use]"
               value="1"
               <?php checked(1, $value); ?> />
        <span class="awp-slider"></span>
    </label>
    <p class="description" style="display:inline-block;margin-left:10px;"><?php _e('Check this box if the coupon cannot be used with other coupons.', 'awp'); ?></p>
    <?php
}


    public function render_coupon_minimum_spend_field() {
        $options = get_option($this->option_name);
        $value = $options['coupon_settings']['minimum_spend'] ?? '';
        echo '<input type="number" name="' . $this->option_name . '[coupon_settings][minimum_spend]" value="' . esc_attr($value) . '" min="0" step="any" placeholder="' . esc_attr__('No minimum', 'awp') . '" style="width: 120px;" />';
    }
public function render_coupon_exclude_sale_items_field() {
    $options = get_option($this->option_name, []);
    $value = (int)($options['coupon_settings']['exclude_sale_items'] ?? 0);
    ?>
    <label class="awp-switch">
        <input type="hidden" name="<?php echo $this->option_name; ?>[coupon_settings][exclude_sale_items]" value="0" />
        <input type="checkbox"
               name="<?php echo $this->option_name; ?>[coupon_settings][exclude_sale_items]"
               value="1"
               <?php checked(1, $value); ?> />
        <span class="awp-slider"></span>
    </label>
    <p class="description" style="display:inline-block;margin-left:10px;"><?php _e('Do not apply this coupon to items on sale.', 'awp'); ?></p>
    <?php
}

    public function sanitize_settings($input) {
            
        $senders_opts = get_option('awp_senders_enabled', ['email' => 1, 'wa' => 1, 'meta' => 1]);
        $is_email_sender_enabled = !empty($senders_opts['email']);
        $is_wa_sender_enabled    = !empty($senders_opts['wa']);
        $is_meta_sender_enabled  = !empty($senders_opts['meta']);
        
        if (isset($input['coupon_settings'])) {
            $coupon_settings = $input['coupon_settings'];
            $sanitized_coupon = [];
            $sanitized_coupon['enable'] = !empty($coupon_settings['enable']) ? 1 : 0;
            $sanitized_coupon['amount'] = isset($coupon_settings['amount']) ? wc_format_decimal($coupon_settings['amount']) : '10';
            $sanitized_coupon['discount_type'] = in_array($coupon_settings['discount_type'], ['percent', 'fixed_cart']) ? $coupon_settings['discount_type'] : 'percent';
            $sanitized_coupon['expiry_value'] = isset($coupon_settings['expiry_value']) ? absint($coupon_settings['expiry_value']) : 7;
            $sanitized_coupon['expiry_unit'] = in_array($coupon_settings['expiry_unit'], ['hours', 'days', 'weeks', 'months', 'years']) ? $coupon_settings['expiry_unit'] : 'days';
            $sanitized_coupon['free_shipping'] = !empty($coupon_settings['free_shipping']) ? 1 : 0;
            $sanitized_coupon['individual_use'] = !empty($coupon_settings['individual_use']) ? 1 : 0;
            $sanitized_coupon['minimum_spend'] = isset($coupon_settings['minimum_spend']) && $coupon_settings['minimum_spend'] !== '' ? wc_format_decimal($coupon_settings['minimum_spend']) : '';
            $sanitized_coupon['exclude_sale_items'] = !empty($coupon_settings['exclude_sale_items']) ? 1 : 0;
            $input['coupon_settings'] = $sanitized_coupon;
        }

        if (isset($input['potential_customer']['notifications']) && is_array($input['potential_customer']['notifications'])) {
        foreach ($input['potential_customer']['notifications'] as $index => &$notification) {
            if (isset($notification['send_via']) && is_array($notification['send_via'])) {
                if (!$is_wa_sender_enabled) {
                    $notification['send_via'] = array_diff($notification['send_via'], ['wawp_whatsapp']);
                }
                if (!$is_meta_sender_enabled) {
                    $notification['send_via'] = array_diff($notification['send_via'], ['meta_whatsapp']);
                }
                if (!$is_email_sender_enabled) {
                    $notification['send_via'] = array_diff($notification['send_via'], ['email']);
                }
            }
        }
        unset($notification); 
            $input['potential_customer']['notifications'] = array_values($input['potential_customer']['notifications']);
        }
        if (isset($input['abandoned_cart']['notifications']) && is_array($input['abandoned_cart']['notifications'])) {
            
        foreach ($input['abandoned_cart']['notifications'] as $index => &$notification) {
            if (isset($notification['send_via']) && is_array($notification['send_via'])) {
                if (!$is_wa_sender_enabled) {
                    $notification['send_via'] = array_diff($notification['send_via'], ['wawp_whatsapp']);
                }
                if (!$is_meta_sender_enabled) {
                    $notification['send_via'] = array_diff($notification['send_via'], ['meta_whatsapp']);
                }
                if (!$is_email_sender_enabled) {
                    $notification['send_via'] = array_diff($notification['send_via'], ['email']);
                }
            }
        }
        unset($notification); 
            $input['abandoned_cart']['notifications'] = array_values($input['abandoned_cart']['notifications']);
        }
        return $input;
    }
    
    public function schedule_notifications_for_cart($cart_id, $type) {
        global $wpdb;
        $log_table = $wpdb->prefix . 'wawp_abandoned_cart_notifications_log';
        $cart_table = $wpdb->prefix . 'wawp_abandoned_carts';

        $existing_count = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM $log_table WHERE cart_id = %d AND notification_type = %s",
            $cart_id,
            $type
        ));

        if ($existing_count > 0) {
            return;
        }

        $this->cancel_all_scheduled_notifications_for_cart($cart_id);

        $options = get_option($this->option_name);
        $notification_sequence = $options[$type]['notifications'] ?? [];

        if (empty($notification_sequence)) {
            return;
        }

        $sequence_uses_coupon = false;
        foreach ($notification_sequence as $settings) {
            if (!empty($settings['enable']) && !empty($settings['use_coupon'])) {
                $sequence_uses_coupon = true;
                break;
            }
        }

        if ($sequence_uses_coupon) {
            $existing_coupon = $wpdb->get_var($wpdb->prepare("SELECT coupon_code FROM $cart_table WHERE id = %d", $cart_id));
            if (empty($existing_coupon)) {
                $coupon_code = $this->generate_coupon_code($cart_id, $options['coupon_settings'] ?? []);
                if (!empty($coupon_code)) {
                    $wpdb->update(
                        $cart_table,
                        ['coupon_code' => $coupon_code],
                        ['id' => $cart_id],
                        ['%s'],
                        ['%d']
                    );
                }
            }
        }

        foreach ($notification_sequence as $index => $settings) {
            if (empty($settings['enable'])) {
                continue;
            }
            
            $send_timing = $settings['send_timing'] ?? 'instant';
            $delay_in_seconds = 0;

            if ($send_timing === 'delayed') {
                $delay_value = (int)($settings['delay_value'] ?? 1);
                $delay_unit = $settings['delay_unit'] ?? 'minutes';
                switch ($delay_unit) {
                    case 'hours': $delay_in_seconds = $delay_value * 3600; break;
                    case 'days': $delay_in_seconds = $delay_value * 86400; break;
                    case 'minutes': default: $delay_in_seconds = $delay_value * 60; break;
                }
            }
            
            $scheduled_time = time() + $delay_in_seconds;

            wp_schedule_single_event($scheduled_time, 'wawp_send_abandoned_cart_notification_step', array($cart_id, $type, $index));
            
            $this->log_internal_notification($cart_id, $type, $index, 'system', 'system', 'scheduled', date('Y-m-d H:i:s', $scheduled_time), null);
        }
    }

    public function send_notification_step($cart_id, $type, $index) {
        require_once AWP_PLUGIN_DIR . 'includes/class-database-manager.php';
        require_once AWP_PLUGIN_DIR . 'includes/class-awp-log-manager.php';
        require_once AWP_PLUGIN_DIR . 'includes/class-wawp-api-url.php';
        require_once AWP_PLUGIN_DIR . 'includes/class-wawp-connector.php';

        global $wpdb;
        $cart_table = $wpdb->prefix . 'wawp_abandoned_carts';
        $cart = $wpdb->get_row($wpdb->prepare("SELECT * FROM $cart_table WHERE id = %d", $cart_id));

        if (!$cart || $cart->status === 'purchased') {
            $this->cancel_all_scheduled_notifications_for_cart($cart_id);
            return;
        }

        $options = get_option($this->option_name);
        $settings = $options[$type]['notifications'][$index] ?? null;

        if (!$settings || empty($settings['enable'])) {
            return;
        }

        $contact_details = maybe_unserialize($cart->contact_details);
        $phone = $contact_details['billing_phone'] ?? null;
        $email = $contact_details['billing_email'] ?? null;
        
        $coupon_code = '';
        if (!empty($settings['use_coupon'])) {
            $coupon_code = $cart->coupon_code ?? '';
        }

        $selected_channels = $settings['send_via'] ?? [];
        if (!is_array($selected_channels)) {
            $selected_channels = [$selected_channels];
        }

        if (in_array('wawp_whatsapp', $selected_channels) && $this->is_wawp_active() && !empty($phone) && !empty($settings['whatsapp_message'])) {
            $global_instance_id = get_option('wawp_abandoned_cart_instance'); 
            if ($global_instance_id && class_exists('Wawp_Api_Url') && class_exists('AWP_Log_Manager') && class_exists('AWP_Database_Manager')) {
                $db_manager = new AWP_Database_Manager();
                $instance = $db_manager->get_instance_by_id($global_instance_id);
                if($instance) {
                    $message = $this->replace_placeholders($settings['whatsapp_message'], $cart, $coupon_code);
                    $response = Wawp_Api_Url::send_message($instance->instance_id, $instance->access_token, $phone, $message);
                    
                    $log_manager = new AWP_Log_Manager();
                    $log_manager->log_notification([
                        'user_id' => $cart->user_id,
                        'order_id' => null,
                        'customer_name' => ($contact_details['billing_first_name'] ?? '') . ' ' . ($contact_details['billing_last_name'] ?? ''),
                        'sent_at' => current_time('mysql'),
                        'whatsapp_number' => $phone,
                        'message' => $message,
                        'message_type' => 'abandoned_cart_' . $type . '_step_' . ($index + 1),
                        'wawp_status' => $response,
                    ]);

                    $this->log_internal_notification($cart_id, $type, $index, 'wawp_whatsapp', $phone, 'sent', null, json_encode($response));
                }
            }
        }

        if (in_array('meta_whatsapp', $selected_channels) && !empty($phone) && !empty($settings['meta_template'])) {
            $this->send_meta_whatsapp_notification($settings['meta_template'], $phone, $cart, $coupon_code, $type, $index);
        }
        
        if (in_array('email', $selected_channels) && !empty($email) && !empty($settings['email_subject']) && !empty($settings['email_body'])) {
            $subject = $this->replace_placeholders($settings['email_subject'], $cart, $coupon_code);
            $body = $this->replace_placeholders($settings['email_body'], $cart, $coupon_code);
            $headers = ['Content-Type: text/html; charset=UTF-8'];
            $sent = wp_mail($email, $subject, $body, $headers);
            $this->log_internal_notification($cart_id, $type, $index, 'email', $email, $sent ? 'sent' : 'failed', null, '');
        }
    }

    private function send_meta_whatsapp_notification($template_info, $phone, $cart, $coupon_code, $type, $index) {
        global $wpdb;
        $db_manager = new AWP_Database_Manager();
        $settings_table = $db_manager->tables['mws_settings'];
        $logs_table = $db_manager->tables['mws_logs'];

        list($template_name, $language_code) = explode('|', $template_info);

        $api_token = $wpdb->get_var($wpdb->prepare("SELECT setting_value FROM " . $settings_table . " WHERE setting_name = %s", 'awp_mws_api_token'));
        $phone_number_id = $wpdb->get_var($wpdb->prepare("SELECT setting_value FROM " . $settings_table . " WHERE setting_name = %s", 'awp_mws_phone_number_id'));

        if (empty($api_token) || empty($phone_number_id)) {
            return;
        }

        $contact_details = maybe_unserialize($cart->contact_details);
        $customer_name = $contact_details['billing_first_name'] ?? 'Valued Customer';
        
        $payload = [
            'messaging_product' => 'whatsapp',
            'to' => $phone,
            'type' => 'template',
            'template' => [
                'name' => $template_name,
                'language' => ['code' => $language_code],
                'components' => [
                    [
                        'type' => 'header',
                        'parameters' => [
                            ['type' => 'text', 'text' => $customer_name]
                        ]
                    ],
                    [
                        'type' => 'body',
                        'parameters' => [
                            ['type' => 'text', 'text' => wc_price($cart->cart_total)],
                            ['type' => 'text', 'text' => $coupon_code],
                        ]
                    ]
                ]
            ]
        ];

        $api_url = "https://graph.facebook.com/v20.0/{$phone_number_id}/messages";
        $response = wp_remote_post($api_url, [
            'headers' => ['Authorization' => 'Bearer ' . $api_token, 'Content-Type' => 'application/json'],
            'body' => json_encode($payload),
            'timeout' => 30
        ]);

        if (!is_wp_error($response)) {
            $body = json_decode(wp_remote_retrieve_body($response), true);
            $message_id = $body['messages'][0]['id'] ?? null;

            if (isset($logs_table)) {
                $wpdb->insert($logs_table, [
                    'message_id' => $message_id,
                    'recipient_number' => $phone,
                    'template_name' => $template_name,
                    'status' => 'sent',
                    'updated_at' => current_time('mysql', 1)
                ]);
            }
            
            $this->log_internal_notification($cart->id, $type, $index, 'meta_whatsapp', $phone, 'sent', null, json_encode($body));
        } else {
            $this->log_internal_notification($cart->id, $type, $index, 'meta_whatsapp', $phone, 'failed', null, $response->get_error_message());
        }
    }

    public function cancel_all_scheduled_notifications_for_cart($cart_id) {
        global $wpdb;
        $log_table = $wpdb->prefix . 'wawp_abandoned_cart_notifications_log';
        
        $scheduled_events = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM $log_table WHERE cart_id = %d AND status = 'scheduled'",
            $cart_id
        ));

        foreach ($scheduled_events as $event) {
            $timestamp = strtotime($event->scheduled_at);
            $hook = 'wawp_send_abandoned_cart_notification_step';
            $args = array($cart_id, $event->notification_type, (int)$event->notification_index);
            
            wp_unschedule_event($timestamp, $hook, $args);

            $wpdb->update(
                $log_table,
                ['status' => 'cancelled'],
                ['id' => $event->id]
            );
        }
    }
    
    private function is_wawp_active() {
        return class_exists('Wawp_Connector');
    }

    private function log_internal_notification($cart_id, $type, $index, $channel, $recipient, $status, $scheduled_at = null, $api_response = '') {
        global $wpdb;
        $table = $wpdb->prefix . 'wawp_abandoned_cart_notifications_log';
        
        $data = [
            'cart_id' => $cart_id,
            'notification_type' => $type,
            'notification_index' => $index,
            'channel' => $channel,
            'recipient' => $recipient,
            'status' => $status,
            'api_response' => $api_response
        ];

        if ($status === 'scheduled') {
            $data['scheduled_at'] = $scheduled_at;
        } else {
            $data['sent_at'] = current_time('mysql');
        }

        $wpdb->insert($table, $data);
    }

    public function generate_coupon_code($cart_id, $settings) {
        if (!function_exists('wc_get_coupon_id_by_code')) {
            return '';
        }

        $coupon_code = 'REC-' . $cart_id . '-' . strtoupper(wp_generate_password(6, false));
        
        if (wc_get_coupon_id_by_code($coupon_code)) {
            $coupon_code = $coupon_code . '-' . rand(10, 99);
        }

        $coupon = new WC_Coupon();
        $coupon->set_code($coupon_code);
        $coupon->set_discount_type($settings['discount_type'] ?? 'percent');
        $coupon->set_amount($settings['amount'] ?? 10);
        
        
        $coupon->set_free_shipping( !empty($settings['free_shipping']) );
        $coupon->set_individual_use( !empty($settings['individual_use']) );
        $coupon->set_minimum_amount( !empty($settings['minimum_spend']) ? floatval($settings['minimum_spend']) : '' );
        $coupon->set_exclude_sale_items( !empty($settings['exclude_sale_items']) );
        
        $coupon->set_usage_limit(1);
        
        $expiry_value = (int)($settings['expiry_value'] ?? 7);
        $expiry_unit = $settings['expiry_unit'] ?? 'days';
        $expiry_date = strtotime("+$expiry_value $expiry_unit");
        $coupon->set_date_expires($expiry_date);
        
        $coupon->save();

        return $coupon_code;
    }

    private function replace_placeholders($template, $cart, $coupon_code = '') {
        $contact_details = maybe_unserialize($cart->contact_details);
        $cart_contents = maybe_unserialize($cart->cart_contents);

        $first_name = $contact_details['billing_first_name'] ?? '';
        $last_name = $contact_details['billing_last_name'] ?? '';
        $full_name = trim($first_name . ' ' . $last_name);
        if (empty($full_name)) {
            $full_name = 'Valued Customer';
        }

        $product_list_html = '<ul>';
        $product_list_text = '';
        if (is_array($cart_contents)) {
            foreach ($cart_contents as $item) {
                $product = wc_get_product($item['variation_id'] ?: $item['product_id']);
                if ($product) {
                    $product_list_html .= sprintf('<li><a href="%s">%s</a> x %d - %s</li>', esc_url($product->get_permalink()), esc_html($product->get_name()), $item['quantity'], wc_price($product->get_price() * $item['quantity']));
                    $product_list_text .= sprintf("%s x %d - %s\n", $product->get_name(), $item['quantity'], wc_price($product->get_price() * $item['quantity']));
                }
            }
        }
        $product_list_html .= '</ul>';

        $replacements = [
            '{first_name}' => $first_name,
            '{last_name}' => $last_name,
            '{name}' => $full_name,
            '{email}' => $contact_details['billing_email'] ?? '',
            '{phone_number}' => $contact_details['billing_phone'] ?? '',
            '{billing_address_1}' => $contact_details['billing_address_1'] ?? '',
            '{billing_city}' => $contact_details['billing_city'] ?? '',
            '{billing_postcode}' => $contact_details['billing_postcode'] ?? '',
            '{billing_country}' => $contact_details['billing_country'] ?? '',
            '{cart_total}' => wc_price($cart->cart_total),
            '{product_in_cart_list_html}' => $product_list_html,
            '{product_in_cart_list}' => trim($product_list_text),
            '{coupon_code}' => $coupon_code,
            '{site_title}' => get_bloginfo('name'),
            '{site_url}' => home_url(),
        ];

        return str_replace(array_keys($replacements), array_values($replacements), $template);
    }

    public function get_all_placeholders() {
        return [
            '{name}' => 'Customer full name',
            '{first_name}' => 'Customer first name',
            '{last_name}' => 'Customer last name',
            '{email}' => 'Customer email',
            '{phone_number}' => 'Customer phone number',
            '{cart_total}' => 'Total value of the cart',
            '{product_in_cart_list}' => 'A plain text list of products in the cart',
            '{product_in_cart_list_html}' => 'An HTML list of products for emails',
            '{coupon_code}' => 'The generated unique coupon code',
            '{site_title}' => 'Your website title',
            '{site_url}' => 'Your website URL',
        ];
    }
    
    
}