<?php
/**
 * This file handles the display and functionality of the Meta WhatsApp Sender logs.
 * It uses WP_List_Table for displaying log data, includes webhook handling for status updates,
 * and a statistics dashboard for a quick overview.
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

if (!defined('AWP_MWS_LOGS_TABLE')) {
    global $wpdb;
    define('AWP_MWS_LOGS_TABLE', $wpdb->prefix . 'awp_mws_message_logs');
}

if (!defined('AWP_MWS_SETTINGS_TABLE')) {
    global $wpdb;
    define('AWP_MWS_SETTINGS_TABLE', $wpdb->prefix . 'awp_mws_settings');
}

// Ensure WP_List_Table is available
if (!class_exists('WP_List_Table')) {
    require_once(ABSPATH . 'wp-admin/includes/class-wp-list-table.php');
}

/**
 * AWP_Logs_List_Table Class
 * * Creates a WP_List_Table for displaying Meta WhatsApp logs.
 */
class AWP_Logs_List_Table extends WP_List_Table {

    /**
     * Constructor.
     */
    public function __construct() {
        parent::__construct([
            'singular' => __('Log Entry', 'awp'),
            'plural'   => 'log_entries', // Nonce actions cannot be translated
            'ajax'     => false
        ]);
    }

    /**
     * Get the list of columns for the log table.
     * @return array The columns for the table.
     */
    public function get_columns() {
        return [
            'cb'               => '<input type="checkbox" />',
            'updated_at'       => __('Last Updated (UTC)', 'awp'),
            'status'           => __('Status', 'awp'),
            'recipient_number' => __('Recipient', 'awp'),
            'template_name'    => __('Content / Template Name', 'awp'),
            'message_id'       => __('Message ID', 'awp'),
        ];
    }

    /**
     * Define which columns are sortable.
     * @return array An associative array of sortable columns.
     */
    protected function get_sortable_columns() {
        return [
            'updated_at'       => ['updated_at', true], // true means it's sorted descending by default
            'status'           => ['status', false],
            'recipient_number' => ['recipient_number', false],
        ];
    }
    
    /**
     * Define the bulk actions.
     * @return array An associative array of bulk actions.
     */
    protected function get_bulk_actions() {
        return [
            'bulk-delete' => __('Delete', 'awp'),
        ];
    }

    /**
     * Process the bulk actions.
     */
    public function process_bulk_action() {
        if ('bulk-delete' === $this->current_action()) {
            check_admin_referer('bulk-' . $this->_args['plural']);
            $log_ids = isset($_REQUEST['log']) ? array_map('intval', $_REQUEST['log']) : [];

            if (!empty($log_ids)) {
                global $wpdb;
                $table_name = AWP_MWS_LOGS_TABLE;
                $ids_placeholder = implode(',', array_fill(0, count($log_ids), '%d'));
                $wpdb->query($wpdb->prepare("DELETE FROM $table_name WHERE id IN ($ids_placeholder)", $log_ids));
                
                add_action('admin_notices', function() {
                    echo '<div class="notice notice-success is-dismissible"><p>' . __('Selected log entries have been deleted.', 'awp') . '</p></div>';
                });
            }
        }
    }

    /**
     * Prepare the items for the table to process by querying the database.
     */
    public function prepare_items() {
        global $wpdb;
        $table_name = AWP_MWS_LOGS_TABLE;

        $this->process_bulk_action();

        $this->_column_headers = [$this->get_columns(), [], $this->get_sortable_columns()];

        $per_page = 25;
        $current_page = $this->get_pagenum();
        $offset = ($current_page - 1) * $per_page;

        $orderby = isset($_REQUEST['orderby']) && in_array($_REQUEST['orderby'], array_keys($this->get_sortable_columns())) ? $_REQUEST['orderby'] : 'updated_at';
        $order = isset($_REQUEST['order']) && in_array(strtoupper($_REQUEST['order']), ['ASC', 'DESC']) ? $_REQUEST['order'] : 'DESC';
        $search_term = isset($_REQUEST['s']) ? sanitize_text_field($_REQUEST['s']) : '';
        
        $where_clauses = [];
        $prepare_args = [];

        if (!empty($search_term)) {
            $where_clauses[] = "(recipient_number LIKE %s OR template_name LIKE %s OR message_id LIKE %s)";
            $like_term = '%' . $wpdb->esc_like($search_term) . '%';
            $prepare_args[] = $like_term;
            $prepare_args[] = $like_term;
            $prepare_args[] = $like_term;
        }
        
        $where_sql = !empty($where_clauses) ? ' WHERE ' . implode(' AND ', $where_clauses) : '';

        // Get the total number of log items.
        $total_items_query = "SELECT COUNT(id) FROM $table_name" . $where_sql;
        
        // FIX: Only use wpdb::prepare when there are parameters to prepare.
        if (!empty($prepare_args)) {
            $total_items = $wpdb->get_var($wpdb->prepare($total_items_query, $prepare_args));
        } else {
            $total_items = $wpdb->get_var($total_items_query);
        }

        $this->set_pagination_args([
            'total_items' => $total_items,
            'per_page'    => $per_page,
            'total_pages' => ceil($total_items / $per_page)
        ]);

        // Fetch the log data from the database.
        $data_query = "SELECT * FROM $table_name" . $where_sql . " ORDER BY $orderby $order LIMIT %d OFFSET %d";
        array_push($prepare_args, $per_page, $offset);
        
        $this->items = $wpdb->get_results($wpdb->prepare($data_query, $prepare_args), ARRAY_A);
    }

    /**
     * Render a column when no custom method exists.
     */
    protected function column_default($item, $column_name) {
        return isset($item[$column_name]) ? esc_html($item[$column_name]) : '';
    }

    /**
     * Render the checkbox column.
     */
    protected function column_cb($item) {
        return sprintf('<input type="checkbox" name="log[]" value="%d" />', $item['id']);
    }
    
    /**
     * Render the status column with styled output.
     */
    protected function column_status($item) {
        $status = strtolower($item['status']);
        $status_colors = [
            'sent'      => '#2271b1', // Blue
            'delivered' => '#00a32a', // Green
            'read'      => '#313a8a', // Dark Blue/Purple
            'failed'    => '#d63638', // Red
            'received'  => '#777777'  // Grey
        ];
        $color = isset($status_colors[$status]) ? $status_colors[$status] : '#777777';
        
        return sprintf(
            '<span class="awp-status %s" style="background-color: %s; color: #fff; padding: 3px 8px; border-radius: 4px; text-transform: capitalize;">%s</span>',
            esc_attr($status),
            esc_attr($color),
            esc_html($item['status'])
        );
    }
    
    /**
     * Render the template_name column, which can contain long text.
     */
    protected function column_template_name($item) {
        return esc_html($item['template_name']);
    }
}

class WAWP_Meta_Log_Plugin {

    /**
     * Constructor. Hooks into WordPress actions.
     */
    public function __construct() {
        add_action('rest_api_init', [$this, 'register_webhook_endpoint']);
    }

    /**
     * Renders the log page content.
     */
    public static function render_log_page() {
        global $wpdb;
        $table_name = AWP_MWS_LOGS_TABLE;
        $log_list_table = new AWP_Logs_List_Table();
        $log_list_table->prepare_items();

        // Calculate statistics
        $total_logs = $wpdb->get_var("SELECT COUNT(id) FROM {$table_name}");
        $sent_logs = $wpdb->get_var("SELECT COUNT(id) FROM {$table_name} WHERE status IN ('sent', 'delivered', 'read')");
        $read_logs = $wpdb->get_var("SELECT COUNT(id) FROM {$table_name} WHERE status = 'read'");
        $pending_logs = $wpdb->get_var("SELECT COUNT(id) FROM {$table_name} WHERE status = 'received'");
        $error_logs = $wpdb->get_var("SELECT COUNT(id) FROM {$table_name} WHERE status = 'failed'");
        $unique_recipients = $wpdb->get_var("SELECT COUNT(DISTINCT recipient_number) FROM {$table_name} WHERE status IN ('sent', 'delivered', 'read')");
        $read_rate = ($sent_logs > 0) ? number_format(($read_logs / $sent_logs) * 100, 2) : 0;
        ?>
            <div class="page-header_row">
                <div class="page-header">
                    <h2 class="page-title"><?php esc_html_e('Meta WhatsApp Messages History', 'awp'); ?></h2>
                    <p><?php esc_html_e('Track all messages sent via Meta API and quickly access info via live search, filters, and logs.', 'awp'); ?></p>
                </div>
            </div>

            <!-- Statistics Cards -->
            <div class="awp-cards" style="margin-bottom: 1.25rem;">
                <div class="whatsapp-logs">
                    <div class="awp-card"><div class="card-header"><h4 class="card-title"><i class="ri-archive-line"></i>Total Logged</h4><p>All entries in current view.</p></div><div class="card_number"><?php echo esc_html($total_logs); ?></div></div>
                    <div class="awp-card status-sent"><div class="card-header"><h4 class="card-title"><i class="ri-whatsapp-line"></i>Notifications Sent</h4><p>Successfully processed & sent.</p></div><div class="card_number"><?php echo esc_html($sent_logs); ?></div></div>
                    <div class="awp-card status-opened"><div class="card-header"><h4 class="card-title"><i class="ri-eye-line"></i>Reads</h4><p><?php echo esc_html($read_rate); ?>% Read Rate (of sent)</p></div><div class="card_number"><?php echo esc_html($read_logs); ?></div></div>
                    <div class="awp-card status-pending"><div class="card-header"><h4 class="card-title"><i class="ri-history-line"></i>Pending</h4><p>Queued for sending.</p></div><div class="card_number"><?php echo esc_html($pending_logs); ?></div></div>
                    <div class="awp-card status-error"><div class="card-header"><h4 class="card-title"><i class="ri-error-warning-line"></i>Send Errors</h4><p>Failed to send.</p></div><div class="card_number"><?php echo esc_html($error_logs); ?></div></div>
                    <div class="awp-card status-users"><div class="card-header"><h4 class="card-title"><i class="ri-user-line"></i>Unique Recipients</h4><p>Unique numbers successfully sent to.</p></div><div class="card_number"><?php echo esc_html($unique_recipients); ?></div></div>
                </div>
            </div>
            
            <form method="post">
                <input type="hidden" name="page" value="<?php echo esc_attr($_REQUEST['page']); ?>" />
                <input type="hidden" name="awp_section" value="<?php echo esc_attr($_REQUEST['awp_section']); ?>" />
                <?php
                    // The search box needs to be inside the form to work correctly.
                    // We wrap it to apply the same styling as the other log page.
                    echo '<div class="wawp-log-filters"><div class="live-search-bar">';
                    $log_list_table->search_box(__('Search Logs', 'awp'), 'log_search_input');
                    echo '</div></div>';
                    
                    $log_list_table->display();
                ?>
            </form>
        <?php
    }

    /**
     * Registers the REST API endpoint for the webhook.
     */
    public function register_webhook_endpoint() {
        register_rest_route('awp/v1', '/webhook', [
            ['methods' => WP_REST_Server::READABLE, 'callback' => [$this, 'handle_webhook_verification'], 'permission_callback' => '__return_true'],
            ['methods' => WP_REST_Server::CREATABLE, 'callback' => [$this, 'handle_webhook_data'], 'permission_callback' => '__return_true'],
        ]);
    }

    /**
     * Handles the webhook verification challenge from Meta.
     */
    public function handle_webhook_verification($request) {
        $verify_token = $this->get_setting('awp_mws_verify_token');
        if (
            isset($request['hub_mode']) && $request['hub_mode'] === 'subscribe' &&
            isset($request['hub_verify_token']) && !empty($verify_token) &&
            $request['hub_verify_token'] === $verify_token
        ) {
            echo esc_html($request['hub_challenge']);
            exit;
        }
        return new WP_Error('invalid_token', 'Invalid verification token', ['status' => 403]);
    }

    /**
     * Handles incoming webhook data (e.g., status updates).
     */
    public function handle_webhook_data($request) {
        global $wpdb;
        $payload = $request->get_body();
        $data = json_decode($payload, true);

        if (isset($data['entry'][0]['changes'][0]['value']['statuses'][0])) {
            $status_data = $data['entry'][0]['changes'][0]['value']['statuses'][0];
            $message_id = $status_data['id'];
            $status = $status_data['status'];

            $wpdb->update(
                AWP_MWS_LOGS_TABLE,
                ['status' => $status, 'updated_at' => current_time('mysql', 1)],
                ['message_id' => $message_id],
                ['%s', '%s'], ['%s']
            );
        }

        return new WP_REST_Response('Success', 200);
    }

    /**
     * Helper function to get a setting from the awp_mws_settings table.
     */
    private function get_setting($name) {
        global $wpdb;
        if ($wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", AWP_MWS_SETTINGS_TABLE)) != AWP_MWS_SETTINGS_TABLE) {
            return '';
        }
        return $wpdb->get_var($wpdb->prepare("SELECT setting_value FROM " . AWP_MWS_SETTINGS_TABLE . " WHERE setting_name = %s", $name));
    }
}

new WAWP_Meta_Log_Plugin();
