<?php
    if (!defined('ABSPATH')) {
        exit;
    }

    if (!defined('AWP_MWS_SETTINGS_TABLE')) {
        global $wpdb;
        define('AWP_MWS_SETTINGS_TABLE', $wpdb->prefix . 'awp_mws_settings');
        define('AWP_MWS_TEMPLATES_TABLE', $wpdb->prefix . 'awp_mws_templates');
    }
    
    add_action( 'phpmailer_init', function ( PHPMailer\PHPMailer\PHPMailer $phpmailer ) {

        $opts = get_option( 'awp_smtp_settings', [] );
        if ( empty( $opts['enabled'] ) ) {
            return;           
        }
    
        $phpmailer->isSMTP();
        $phpmailer->Host       = $opts['host']       ?? '';
        $phpmailer->Port       = (int) ( $opts['port'] ?? 587 );
        $encryption           = $opts['encryption']   ?? 'tls';  
        $phpmailer->SMTPSecure = ( $encryption === 'none' ? '' : $encryption );
        $phpmailer->SMTPAuth   = ! empty( $opts['auth'] );
        if ( $phpmailer->SMTPAuth ) {
            $phpmailer->Username = $opts['user'] ?? '';
            $phpmailer->Password = $opts['pass'] ?? '';
        }
        $phpmailer->setFrom(
            $opts['from_email'] ?? get_bloginfo( 'admin_email' ),
            $opts['from_name']  ?? get_bloginfo( 'name' )
        );
    }, 20 );
    
    class AWP_Instances {
        private $database_manager;
        private $wawp_domain = 'https://wawp.net';
    
        public function __construct() {
            $this->database_manager = new AWP_Database_Manager();
        }
    
        public function init() {
            add_action('wp_ajax_awp_add_instance', [$this, 'add_instance']);
            add_action('wp_ajax_awp_delete_instance', [$this, 'delete_instance']);
            add_action('wp_ajax_awp_edit_instance', [$this, 'edit_instance']);
            add_action('wp_ajax_awp_update_status', [$this, 'update_status']);
            add_action('wp_ajax_awp_send_test_message', [$this, 'send_test_message']);
            add_action('admin_notices', [$this, 'admin_notices']);
            add_action('wp_ajax_awp_get_all_instances', [$this, 'awp_get_all_instances_ajax']);
            add_action('wp_ajax_awp_auto_check_all_instance_status', [$this, 'auto_check_all_instance_status']);
            add_action( 'wp_ajax_awp_smtp_send_test_email', [ $this, 'smtp_send_test_email' ] );
            add_action( 'wp_ajax_awp_smtp_test_connection', [ $this, 'smtp_test_connection' ] );
            add_action( 'wp_ajax_awp_ajax_save_senders_settings', [ $this, 'ajax_save_senders_settings' ] );
            add_action('admin_init', [$this, 'handle_meta_sender_forms']);
            add_action( 'wp_ajax_awp_save_block_list', function () {
    
                if ( ! current_user_can( 'manage_options' ) ) {
                    wp_send_json_error( 'unauthorized', 403 );
                }
                check_ajax_referer( 'awp_block_list_nonce', 'nonce' );
            
                $raw  = wp_unslash( $_POST['list'] ?? '' );
                $list = array_filter( array_map( 'trim', explode( "\n", $raw ) ) );
            
                ( new AWP_Database_Manager() )->upsert_blocked_numbers( $list );
            
                wp_send_json_success( [ 'saved' => count( $list ) ] );
            } );
        }
    
        public function render_admin_page($user_data = []) {
        
            if ( ! current_user_can( 'manage_options' ) ) {
                return;
            }
            
            
            $this->maybe_save_block_list();
            
            $user_data = get_transient( 'siteB_user_data' );
            if ( ! is_array( $user_data ) ) {
                $user_data = [];
            }
            
            $block_text = implode("\n", ( new AWP_Database_Manager() )->get_blocked_numbers() );
            $senders_opts = get_option('awp_senders_enabled', ['email' => 1, 'wa' => 1, 'meta' => 1, 'block' => 1]);
            $emailsmtp_allowed    = ( $user_data['emailsmtp_enabled']    ?? 'disabled' ) === 'enabled';
            $wawpsender_allowed   = ( $user_data['wawpsender_enabled']   ?? 'disabled' ) === 'enabled';
            $metasender_allowed   = ( $user_data['metasender_enabled']   ?? 'disabled' ) === 'enabled';
            $blockmanager_allowed = ( $user_data['blockmanager_enabled'] ?? 'disabled' ) === 'enabled';
            $icons_url = AWP_PLUGIN_URL . 'assets/icons/senders/';
            $senders_icon_url = $icons_url . 'radio.svg';
            $email_icon   = $icons_url . 'mail.svg';
            $wa_icon      = $icons_url . 'whatsapp-line.svg';
            $meta_icon    = $icons_url . 'meta-line.svg';
            $block_icon   = $icons_url . 'shield-ban.svg';
            $email_icon_url   = $icons_url . 'Email Sender (SMTP).svg';
            $wa_icon_url      = $icons_url . 'WhatsApp Sender.svg';
            $meta_icon_url    = $icons_url . 'Meta Sender.svg';
            $block_icon_url   = $icons_url . 'Block Manager.svg';
            
            $access_token = $user_data['access_token'] ?? '';
            echo '<script>var awp_main_access_token = "' . esc_js($access_token) . '";</script>';
            ?>
    
            <div class="wrap awp-senders-wrapper">
    
                <div class="page-header_row">
                    <div class="page-header">
                        <h2 class="page-title"><?php _e( 'Sender Settings', 'awp' ); ?></h2>
                        <p>
                            <?php _e( 'Connect your messaging and email channels, and manage blocked numbers.', 'awp' ); ?>
                        </p>
                        <div class="awp-admin-notice-global"></div>
                    </div>
                </div>
    
                <div class="nav-tab-wrapper" id="awp-sender-tabs" style="margin-bottom: 1.25rem;">
                    <a href="#tab-senders" class="nav-tab"><img src="<?php echo esc_url($senders_icon_url); ?>" class="awp-tab-icon" alt=""><?php _e( 'Senders', 'awp' ); ?></a>
                    <a href="#tab-wa"    class="nav-tab" style="<?php echo empty($senders_opts['wa']) ? 'display:none;' : ''; ?>"><img src="<?php echo esc_url($wa_icon); ?>" class="awp-tab-icon" alt=""><?php _e( 'WhatsApp Web', 'awp' ); ?></a>
                    <a href="#tab-meta"  class="nav-tab" style="<?php echo empty($senders_opts['meta']) ? 'display:none;' : ''; ?>"><img src="<?php echo esc_url($meta_icon); ?>" class="awp-tab-icon" alt=""><?php _e( 'Meta API', 'awp' ); ?></a>
                    <a href="#tab-email" class="nav-tab" style="<?php echo empty($senders_opts['email']) ? 'display:none;' : ''; ?>"><img src="<?php echo esc_url($email_icon); ?>" class="awp-tab-icon" alt=""><?php _e( 'Email SMTP', 'awp' ); ?></a>
                    <a href="#tab-block" class="nav-tab" style="<?php echo empty($senders_opts['block']) ? 'display:none;' : ''; ?>"><img src="<?php echo esc_url($block_icon); ?>" class="awp-tab-icon" alt=""><?php _e( 'Block Manager', 'awp' ); ?></a>
                </div>
    
                <div class="awp-tab-content">
                    
                    <div id="tab-senders" class="awp-tab-pane">
                        <form method="post" id="awp-senders-form" action="">
                            <?php wp_nonce_field('awp_save_senders', 'awp_save_senders_nonce'); ?>
                                <div class="awp-senders_section">
                                    <div class="awp-card">
                                        <div class="card-header">
                                            <div class="feature_enabling">
                                                <img src="<?php echo esc_url($wa_icon_url); ?>" class="feature_image">
                                                <label class="awp-switch">
                                                    <input type="checkbox" id="awp_sender_toggle_wa" name="awp_senders_enabled[wa]" value="1" <?php checked($senders_opts['wa'] ?? 1, 1); ?> <?php echo $wawpsender_allowed ? '' : 'disabled'; ?>>
                                                    <span class="custom-control-label"></span>
                                                </label>
                                            </div>
                                            <h4 class="card-title"><?php _e('WhatsApp Web', 'awp'); ?></h4>
                                            <p><?php _e('Connect your personal WhatsApp for automated orders notifications, OTPs, and more.', 'awp'); ?></p>
                                            <div class="awp-card_actions">
                                                <a href="https://help.wawp.net/en_US/sender-settings-tutorial/#whatsapp-web-settings" class="awp-btn" target="_blank">
                                                    <i class="ri-book-open-line"></i>
                                                    <span><?php esc_html_e( 'Docs', 'awp' ); ?></span>
                                                </a>
                                                <a href="#tab-wa" class="awp-btn" style="<?php echo empty($senders_opts['wa']) ? 'display:none;' : ''; ?>">
                                                    <i class="ri-settings-line"></i>
                                                    <span><?php esc_html_e( 'Settings', 'awp' ); ?></span>
                                                </a>
                                                <?php if ( ! $wawpsender_allowed ) : ?>
                                                    <a href="https://wawp.net/product/whatsapp-sender/" target="_blank" class="awp-btn golden-btn"><?php _e( 'Get Addon', 'awp' ); ?></a>
                                                <?php endif; ?>
                                            </div>
                                        </div>
                                    </div> 

                                    <div class="awp-card">
                                        <div class="card-header">
                                            <div class="feature_enabling">
                                                <img src="<?php echo esc_url($meta_icon_url); ?>" class="feature_image">
                                                <?php if ( $metasender_allowed ) : ?>
                                                <label class="awp-switch">
                                                    <input type="checkbox" id="awp_sender_toggle_meta" name="awp_senders_enabled[meta]" value="1" <?php checked($senders_opts['meta'] ?? 1, 1); ?> <?php echo $metasender_allowed ? '' : 'disabled'; ?>>
                                                    <span class="custom-control-label"></span>
                                                </label>
                                                <?php endif; ?>
                                            </div>
                                            <h4 class="card-title"><?php _e('WhatsApp Business API', 'awp'); ?></h4>
                                            <p><?php _e('Official Meta API for scalable business messaging. Ideal for high-volume automations.', 'awp'); ?></p>
                                            <div class="awp-card_actions">
                                                <a href="https://help.wawp.net/en_US/sender-settings-tutorial/#meta-api-settings" class="awp-btn" target="_blank">
                                                    <i class="ri-book-open-line"></i>
                                                    <span><?php esc_html_e( 'Docs', 'awp' ); ?></span>
                                                </a>
                                                <a href="#tab-meta" class="awp-btn" style="<?php echo empty($senders_opts['meta']) ? 'display:none;' : ''; ?>">
                                                    <i class="ri-settings-line"></i>
                                                    <span><?php esc_html_e( 'Settings', 'awp' ); ?></span>
                                                </a>
                                                <?php if ( ! $metasender_allowed ) : ?>
                                                    <a href="https://wawp.net/product/meta-sender/" target="_blank" class="awp-btn golden-btn"><?php _e( 'Get Addon', 'awp' ); ?></a>
                                                <?php endif; ?>
                                            </div>
                                        </div>
                                    </div> 

                                    <div class="awp-card">
                                        <div class="card-header">
                                            <div class="feature_enabling">
                                                <img src="<?php echo esc_url($email_icon_url); ?>" class="feature_image">
                                                <label class="awp-switch">
                                                    <input type="checkbox" id="awp_sender_toggle_email" name="awp_senders_enabled[email]" value="1" <?php checked($senders_opts['email'] ?? 1, 1); ?> <?php echo $emailsmtp_allowed ? '' : 'disabled'; ?>>
                                                    <span class="custom-control-label"></span>
                                                </label>
                                            </div>
                                            <h4 class="card-title"><?php _e('Email Sender (SMTP)', 'awp'); ?></h4>
                                            <p><?php _e('Send transactional and marketing emails via your custom SMTP server for reliability and branding.', 'awp'); ?></p>
                                            <div class="awp-card_actions">
                                                <a href="https://help.wawp.net/en_US/sender-settings-tutorial/#smtp-settings" class="awp-btn" target="_blank">
                                                    <i class="ri-book-open-line"></i>
                                                    <span><?php esc_html_e( 'Docs', 'awp' ); ?></span>
                                                </a>
                                                <a href="#tab-email" class="awp-btn" style="<?php echo empty($senders_opts['email']) ? 'display:none;' : ''; ?>">
                                                    <i class="ri-settings-line"></i>
                                                    <span><?php esc_html_e( 'Settings', 'awp' ); ?></span>
                                                </a>
                                                <?php if ( ! $emailsmtp_allowed ) : ?>
                                                    <a href="https://wawp.net/product/email-sender-smtp/" target="_blank" class="awp-btn golden-btn"><?php _e( 'Get Addon', 'awp' ); ?></a>
                                                <?php endif; ?>
                                            </div>
                                        </div>
                                    </div> 

                                    <div class="awp-card">
                                        <div class="card-header">
                                            <div class="feature_enabling">
                                                <img src="<?php echo esc_url($block_icon_url); ?>" class="feature_image">
                                                <label class="awp-switch">
                                                    <input type="checkbox" id="awp_sender_toggle_block" name="awp_senders_enabled[block]" value="1" <?php checked($senders_opts['block'] ?? 1, 1); ?> <?php echo $blockmanager_allowed ? '' : 'disabled'; ?>>
                                                    <span class="custom-control-label"></span>
                                                </label>
                                            </div>
                                            <h4 class="card-title"><?php _e('Block Manager', 'awp'); ?></h4>
                                            <p><?php _e("Manage and review phone numbers you've blocked to prevent unwanted interactions.", "awp"); ?></p>
                                            <div class="awp-card_actions">
                                                <a href="https://help.wawp.net/en_US/sender-settings-tutorial/#blocked-numbers" class="awp-btn" target="_blank">
                                                    <i class="ri-book-open-line"></i>
                                                    <span><?php esc_html_e( 'Docs', 'awp' ); ?></span>
                                                </a>
                                                <a href="#tab-block" class="awp-btn" style="<?php echo empty($senders_opts['block']) ? 'display:none;' : ''; ?>">
                                                    <i class="ri-settings-line"></i>
                                                    <span><?php esc_html_e( 'Settings', 'awp' ); ?></span>
                                                </a>
                                                <?php if ( ! $blockmanager_allowed ) : ?>
                                                    <a href="https://wawp.net/product/block-manager/" target="_blank" class="awp-btn golden-btn"><?php _e( 'Get Addon', 'awp' ); ?></a>
                                                <?php endif; ?>
                                            </div>
                                        </div>
                                    </div> 
                                </div>
                        </form>
                    </div>

                    <div id="tab-email" class="awp-tab-pane">
                        <?php
                            $this->maybe_save_smtp_settings();
                            $this->render_smtp_settings_form();
                        ?>
    
                    </div>
                    <div id="tab-wa" class="awp-tab-pane">
                            <div id="awp-add-modal" class="awp-modal">
                                <div class="awp-modal-content">
                                    <h4 class="card-title"><i class="ri-whatsapp-line"></i> <?php _e( 'Add New Instance Manually', 'awp' ); ?></h4>
    
                                    <form id="awp-add-instance-form">
                                        <div class="awp-form-group">
                                            <label><?php _e( 'Name', 'awp' ); ?></label>
                                            <input type="text" id="awp-name" placeholder="<?php esc_attr_e( 'e.g., Main Business Line', 'awp' ); ?>">
                                        </div>
                                        <div class="awp-form-group">
                                            <label><?php _e( 'Instance ID', 'awp' ); ?></label>
                                            <input type="text" id="awp-instance-id">
                                        </div>
                                        <div class="awp-form-group">
                                            <label><?php _e( 'Access Token', 'awp' ); ?></label>
                                            <input type="text" id="awp-access-token">
                                        </div>
    
                                        <div class="btn-group" style="margin-top:12px;">
                                            <button type="button" class="awp-btn" id="awp-close-add-modal" style="flex:1;"><?php _e( 'Cancel', 'awp' ); ?></button>
                                            <button type="button" class="awp-btn primary" id="awp-save-add-btn" style="flex:1;"><?php _e( 'Save', 'awp' ); ?></button>
                                        </div>
                                    </form>
                                </div>
                            </div>
    
                            <div id="awp-edit-modal" class="awp-modal">
                                <div class="awp-modal-content">
                                    <h4 class="card-title"><i class="ri-edit-line"></i> <?php _e( 'Edit Instance', 'awp' ); ?></h4>
    
                                    <form id="awp-edit-instance-form">
                                        <input type="hidden" id="edit-id">
    
                                        <div class="awp-form-group">
                                            <label><?php _e( 'Name', 'awp' ); ?></label>
                                            <input type="text" id="edit-name">
                                        </div>
                                        <div class="awp-form-group">
                                            <label><?php _e( 'Instance ID', 'awp' ); ?></label>
                                            <input type="text" id="edit-instance-id">
                                        </div>
                                        <div class="awp-form-group">
                                            <label><?php _e( 'Access Token', 'awp' ); ?></label>
                                            <input type="text" id="edit-access-token">
                                        </div>
    
                                        <div class="btn-group" style="margin-top:12px;">
                                            <button type="button" class="awp-btn" id="awp-close-edit-modal" style="flex:1;"><?php _e( 'Cancel', 'awp' ); ?></button>
                                            <button type="button" class="awp-btn awp-save primary" id="awp-save-edit-btn" style="flex:1;"><?php _e( 'Save Changes', 'awp' ); ?></button>
                                        </div>
                                    </form>
                                </div>
                            </div>
                            
                            <!-- START: New Modals -->
                            <div id="awp-scan-qr-modal" class="awp-modal">
                                <div class="awp-modal-content">
                                    <h4 class="card-title"><i class="ri-qr-scan-2-line"></i><?php _e('Scan QR Code to Connect', 'awp'); ?></h4>
                                    <div id="awp-qr-code-container" style="text-align: center; margin: 20px 0; min-height: 200px; display: flex; align-items: center; justify-content: center;">
                                        <div class="awp-spinner"></div>
                                        <img id="awp-qr-image" src="" alt="QR Code" style="display: none; max-width: 100%; height: auto; border: 1px solid #eee;">
                                    </div>
                                    <p id="awp-qr-status-text" style="text-align: center; min-height: 20px;"></p>
                                    <div class="btn-group" style="margin-top:12px;">
                                        <button type="button" class="awp-btn" id="awp-close-qr-modal" style="flex:1;"><?php _e('Cancel', 'awp'); ?></button>
                                    </div>
                                </div>
                            </div>

                            <div id="awp-connect-by-code-modal" class="awp-modal">
                                <div class="awp-modal-content">
                                    <h4 class="card-title"><i class="ri-hashtag"></i> <?php _e('Connect using Phone Number', 'awp'); ?></h4>
                                    <div id="awp-code-form-container" style="margin-top: .5rem;">
                                        <p style="margin-bottom: .5rem !important;"><?php _e('Enter the phone number associated with your WhatsApp account to receive a connection code on your device.', 'awp'); ?></p>
                                        <div class="awp-form-group">
                                            <label><?php _e('Phone Number', 'awp'); ?></label>
                                            <input type="tel" id="awp-connect-phone-number" placeholder="<?php esc_attr_e('e.g., 447441429009', 'awp'); ?>">
                                        </div>
                                        <div class="btn-group" style="margin-top:12px;">
                                            <button type="button" class="awp-btn" id="awp-close-code-modal" style="flex:1;"><?php _e('Cancel', 'awp'); ?></button>
                                            <button type="button" class="awp-btn primary" id="awp-request-code-btn" style="flex:1;"><?php _e('Request Code', 'awp'); ?></button>
                                        </div>
                                    </div>
                                    <div id="awp-code-status-container" style="display: none; text-align: center; min-height: 100px; justify-content: center; flex-direction: column;">
                                         <div class="awp-spinner"></div>
                                         <p id="awp-code-status-text"></p>
                                    </div>
                                </div>
                            </div>
                             <!-- END: New Modals -->
    
                            <div class="awp-card" style="margin-bottom: 1.25rem;">
                                <div class="card-header_row">
                                    <div class="card-header">
                                        <h4 class="card-title"><?php _e( 'Add your WhatsApp', 'awp' ); ?></h4>
                                        <p><?php
                                        printf(
                                            esc_html__( 'Add the access token and instance ID from %1$sWawp.net%2$s.', 'awp' ),
                                            '<a href="' . esc_url( 'https://wawp.net/account/connect' ) . '" target="_blank" rel="noopener noreferrer">',
                                            '</a>'); ?>
                                        </p>
                                    </div>
                                </div>
                                    <div class="btn-group" style="gap: .75rem;">
                                        <?php
                                          $ud = is_array($user_data) ? $user_data : [];
                                          $instances_limit = null;
                                        
                                          if (isset($ud['instances_limit'])) {
                                              $instances_limit = (int) $ud['instances_limit'];
                                          } elseif (isset($ud['max_instances'])) {
                                              $instances_limit = (int) $ud['max_instances'];
                                          }
                                        
                                          if ($instances_limit === null) {
                                              $is_lifetime = !empty($ud['is_lifetime']);
                                              $has_active  = false;
                                              if (!empty($ud['subscriptions']) && is_array($ud['subscriptions'])) {
                                                  foreach ($ud['subscriptions'] as $sub) {
                                                      if (!empty($sub['status']) && $sub['status'] === 'active') {
                                                          $has_active = true; break;
                                                      }
                                                  }
                                              }
                                              $instances_limit = ($is_lifetime || $has_active) ? 10 : 1;
                                          }
                                        
                                          $all_instances = $this->get_all_instances();
                                          $used_count    = is_array($all_instances) ? count($all_instances) : 0;
                                          $remaining     = max(0, $instances_limit - $used_count);
                                        ?>
                                        <?php if ($remaining > 0) : ?>
                                            <a style="color: #fff;" href="https://wawp.net/account/connect/" target=_blank ><button type="button" class="sender_method_card secondary" id="open-wawp-net">
                                                <i class="ri-qr-scan-2-line"></i><?php _e( 'Link your number', 'awp' ); ?></a>
                                            </button>
                                            <button type="button" class="sender_method_card secondary" id="awp-open-add-modal">
                                                <i class="ri-add-line"></i><?php _e( 'Add Instance Here', 'awp' ); ?>
                                            </button>
                                        <?php else : ?>
                                            <span id="awp-limit-reached" class="awp-badge sender_method_card" style="background:#ef4444;color:#fff;">
                                                <?php _e('Instance limit reached.', 'awp'); ?>
                                            </span>
                                        <?php endif; ?>
                                    </div>
                            </div>
                            
                            <div class="awp-table-container-wrapper">
                                <div style="position:relative;">
                                    <div class="awp-table-container awp-card" style="padding:0;">
    
                                        <div class="card-header_row" style="padding:1.25rem 1.25rem 0;">
                                            <div class="card-header">
                                                <h4 class="card-title"><i class="ri-links-line"></i> <?php _e( 'Connected Numbers', 'awp' ); ?></h4>
                                                <?php
                                                 
                                                  $instances_limit = null;
                                                  $ud = is_array($user_data) ? $user_data : [];
                                                
                                                  if (isset($ud['instances_limit'])) {
                                                      $instances_limit = (int) $ud['instances_limit'];
                                                  } elseif (isset($ud['max_instances'])) {
                                                      $instances_limit = (int) $ud['max_instances'];
                                                  }
                                                  if ($instances_limit === null) {
                                                      $is_lifetime = !empty($ud['is_lifetime']);
                                                      $has_active  = false;
                                                      if (!empty($ud['subscriptions']) && is_array($ud['subscriptions'])) {
                                                          foreach ($ud['subscriptions'] as $sub) {
                                                              if (!empty($sub['status']) && $sub['status'] === 'active') {
                                                                  $has_active = true; break;
                                                              }
                                                          }
                                                      }
                                                      $instances_limit = ($is_lifetime || $has_active) ? 10 : 1; 
                                                  }
                                                
                                                  
                                                  $all_instances = $this->get_all_instances();
                                                  $used_count    = is_array($all_instances) ? count($all_instances) : 0;
                                                  $remaining     = max(0, $instances_limit - $used_count);
                                                
                                                 
                                                  $has_active_or_lifetime = !empty($ud['is_lifetime']);
                                                  if (!$has_active_or_lifetime && !empty($ud['subscriptions']) && is_array($ud['subscriptions'])) {
                                                      foreach ($ud['subscriptions'] as $sub) {
                                                          if (!empty($sub['status']) && $sub['status'] === 'active') {
                                                              $has_active_or_lifetime = true; break;
                                                          }
                                                      }
                                                  }
                                                  $subscription_url = $has_active_or_lifetime
                                                      ? 'https://wawp.net/account/subscription'
                                                      : 'https://wawp.net/pricing';
                                                  $used = $used_count; 
                                                  $quota_html = sprintf(
                                                      _n(
                                                          '%1$d of %2$d number is connected. See your %3$sSubscription%4$s.',
                                                          '%1$d of %2$d numbers are connected. See your %3$sSubscription%4$s.',
                                                          $instances_limit,
                                                          'awp'
                                                      ),
                                                      $used,
                                                      $instances_limit,
                                                      '<a href="' . esc_url($subscription_url) . '" target="_blank" rel="noopener noreferrer">',
                                                      '</a>'
                                                  );
                                                
                                                ?>
                                                <p id="awp-instance-quota"
                                                     data-limit="<?php echo esc_attr($instances_limit); ?>"
                                                     data-subscription-url="<?php echo esc_attr($subscription_url); ?>">
                                                  <?php echo $quota_html; ?>
                                                </p>
                                            </div>
                                        </div>
                                        




                                        <table class="awp-table">
                                            <thead><tr>
                                                <th><?php _e( 'Name',               'awp' ); ?></th>
                                                <th><?php _e( 'Instance ID',        'awp' ); ?></th>
                                                <th><?php _e( 'Access Token',     'awp' ); ?></th>
                                                <th><?php _e( 'Connection Status', 'awp' ); ?></th>
                                                <th><?php _e( 'Actions',            'awp' ); ?></th>
                                            </tr></thead>
                                            <tbody id="awp-table-body"></tbody>
                                        </table>
                                    </div>
                                </div>
                            </div>
    
                            <?php $this->render_otp_senders_form(); ?>
    
                    </div>

                    <div id="tab-meta" class="awp-tab-pane">
                        <div class="awp-card">
                            <div class="card-header">
                                <h4 class="card-title"><?php esc_html_e('Meta API Settings', 'awp'); ?></h4>
                                <p><?php esc_html_e('Configure your Meta API credentials for sending messages and setting up webhooks.', 'awp'); ?></p>
                            </div>
                            <form method="post" action="">
                                <?php wp_nonce_field('awp_mws_save_settings_action', 'awp_mws_save_settings_nonce'); ?>
                            
                                <div class="instances-setup">
                            
                                    <div class="awp-field">
                                        <label for="awp_mws_api_token"><?php esc_html_e('Access Token', 'awp'); ?></label>
                                        <div class="awp-control">
                                            <input type="password" id="awp_mws_api_token" name="awp_mws_api_token" value="<?php echo esc_attr($this->get_setting('awp_mws_api_token')); ?>" class="regular-text" />
                                        </div>
                                    </div>
                            
                                    <div class="awp-field">
                                        <label for="awp_mws_waba_id"><?php esc_html_e('WhatsApp Business Account ID', 'awp'); ?></label>
                                        <div class="awp-control">
                                            <input type="text" id="awp_mws_waba_id" name="awp_mws_waba_id" value="<?php echo esc_attr($this->get_setting('awp_mws_waba_id')); ?>" class="regular-text" />
                                            <p class="description"><?php esc_html_e('Required to fetch message templates.', 'awp'); ?></p>
                                        </div>
                                    </div>
                            
                                    <div class="awp-field">
                                        <label for="awp_mws_phone_number_id"><?php esc_html_e('Phone Number ID', 'awp'); ?></label>
                                        <div class="awp-control">
                                            <input type="text" id="awp_mws_phone_number_id" name="awp_mws_phone_number_id" value="<?php echo esc_attr($this->get_setting('awp_mws_phone_number_id')); ?>" class="regular-text" />
                                            <p class="description"><?php esc_html_e('Required to send messages.', 'awp'); ?></p>
                                        </div>
                                    </div>
                                    <hr class="h-divider">
                                    <div class="awp-full">
                                        <div class="card-header">
                                            <h4 class="card-title"><?php esc_html_e('Webhook Settings', 'awp'); ?></h4>
                                            <p><?php esc_html_e('Set up webhooks to receive real-time events from Meta.', 'awp'); ?></p>
                                        </div>
                                    </div>
                            
                                    <div class="awp-field">
                                        <label for="awp_mws_app_id"><?php esc_html_e('Meta App ID (Client ID)', 'awp'); ?></label>
                                        <div class="awp-control">
                                            <input type="text" id="awp_mws_app_id" name="awp_mws_app_id" value="<?php echo esc_attr($this->get_setting('awp_mws_app_id')); ?>" class="regular-text" />
                                        </div>
                                    </div>
                            
                                    <div class="awp-field">
                                        <label for="awp_mws_app_secret"><?php esc_html_e('Meta App Secret (Client Secret)', 'awp'); ?></label>
                                        <div class="awp-control">
                                            <input type="password" id="awp_mws_app_secret" name="awp_mws_app_secret" value="<?php echo esc_attr($this->get_setting('awp_mws_app_secret')); ?>" class="regular-text" />
                                        </div>
                                    </div>
                            
                                    <div class="awp-field">
                                        <label for="awp_mws_verify_token"><?php esc_html_e('Webhook Verify Token', 'awp'); ?></label>
                                        <div class="awp-control">
                                            <input type="text" id="awp_mws_verify_token" name="awp_mws_verify_token" value="<?php echo esc_attr($this->get_setting('awp_mws_verify_token')); ?>" class="regular-text" />
                                            <p class="description"><?php esc_html_e('Leave blank to auto-generate a secure token when setting the webhook.', 'awp'); ?></p>
                                        </div>
                                    </div>
                            
                                    <div class="awp-field">
                                        <span class="awp-label"><?php esc_html_e('Webhook URL', 'awp'); ?></span>
                                        <div class="awp-control">
                                            <code><?php echo esc_url( rest_url('awp/v1/webhook') ); ?></code>
                                            <p class="description"><?php esc_html_e('This URL will be configured automatically when you click the "Set Webhook" button below.', 'awp'); ?></p>
                                        </div>
                                    </div>
                            
                                    <div class="btn-group">
                                        <?php submit_button(__('Save Settings', 'awp'), 'awp-btn', 'save_settings', false); ?>
                                        <?php submit_button(__('Set Webhook Automatically', 'awp'), 'secondary', 'set_webhook', false); ?>
                                        <?php submit_button(__('Refresh Templates List', 'awp'), 'secondary', 'fetch_templates', false); ?>
                                    </div>
                                    <p><?php esc_html_e('Clicking "Set Webhook" or "Refresh Templates" will also save any changes made to the settings above.', 'awp'); ?></p>

                                </div>
                            </form>
                        </div>
                        <?php
                        global $wpdb;
                        $templates = $wpdb->get_results("SELECT template_name, language_code FROM " . AWP_MWS_TEMPLATES_TABLE . " ORDER BY template_name ASC, language_code ASC");
                    ?>
                        <div class="awp-card" style="margin-top: 1.25rem;">
                            <div class="card-header">
                            <h4 class="card-title"><?php esc_html_e('Available Templates', 'awp'); ?></h4>
                            <?php if (!empty($templates)) : ?>
                                <p><?php esc_html_e('The following templates were successfully fetched from your WhatsApp Business Account.', 'awp'); ?></p>
                                </div>
                                <table class="wp-list-table widefat striped">
                                    <thead>
                                        <tr>
                                            <th scope="col"><?php esc_html_e('Template Name', 'awp'); ?></th>
                                            <th scope="col"><?php esc_html_e('Language Code', 'awp'); ?></th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($templates as $template) : ?>
                                            <tr>
                                                <td><?php echo esc_html($template->template_name); ?></td>
                                                <td><?php echo esc_html($template->language_code); ?></td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            <?php else : ?>
                                <p><?php esc_html_e('No templates found in the database. Please save your settings and click "Refresh Templates List" to fetch them.', 'awp'); ?></p>
                                </div>
                            <?php endif; ?>
                        </div>
                    </div>

                    <div id="tab-block" class="awp-tab-pane">
                        <div class="awp-card">
    
                            <div class="card-header_row">
                                <div class="card-header">
                                    <h4 class="card-title"><?php _e( 'Blocked Numbers', 'awp' ); ?></h4>
                                    <p><?php _e( 'Blocked numbers will no longer be able to login, signup, purchase, or receive messages from your website.', 'awp' ); ?></p>
                                </div>
                            </div>
    
                            <div class="awp-intl-wrap" style="display: flex;align-items: center;gap: .5rem;">
                                <input id="awp_block_intl" type="tel" placeholder="<?php esc_attr_e( 'Type number', 'awp' ); ?>">
                                <button type="button" id="awp_block_add_btn" class="awp-btn secondary" style="height: 39px;"><?php _e( 'Add', 'awp' ); ?></button>
                            </div>
    
                            <form method="post" action="" style="gap: 0;">
                                <?php wp_nonce_field( 'awp_block_list_nonce', 'awp_block_list_nonce_field' ); ?>
    
                                <textarea class="awp-block-area" name="awp_block_list" rows="5" style="display:none;"><?php echo esc_textarea( $block_text ); ?></textarea>
    
                                <input id="awp_block_tagify" type="text" style="width:100%;" placeholder="<?php esc_attr_e( 'Enter phone numbers', 'awp' ); ?>">
                                <p class="submit awp_save">
                                    <button type="submit" name="awp_block_list_submit" class="awp-btn primary" style="height: 39px;padding: .5rem 1rem;"><?php _e( 'Save Block List', 'awp' ); ?></button>
                                </p>
                            </form>
    
                        </div>
                    </div>
    
                </div>
            </div>
            <?php
        }
    
        private function maybe_save_senders_settings() {
            if (
                isset( $_POST['awp_save_senders'] ) &&
                check_admin_referer( 'awp_save_senders', 'awp_save_senders_nonce' )
            ) {
                $raw = $_POST['awp_senders_enabled'] ?? [];
                $clean = [
                    'email'      => isset( $raw['email'] ) ? 1 : 0,
                    'wa'         => isset( $raw['wa'] ) ? 1 : 0,
                    'meta'       => isset( $raw['meta'] ) ? 1 : 0,
                    'block'      => isset( $raw['block'] ) ? 1 : 0,
                ];
                update_option( 'awp_senders_enabled', $clean );
            }
        }

        public function ajax_save_senders_settings() {
            if ( ! current_user_can( 'manage_options' ) || ! check_ajax_referer( 'awp_save_senders', 'awp_save_senders_nonce' ) ) {
                wp_send_json_error( [ 'message' => __( 'Security check failed.', 'awp' ) ], 403 );
            }
        
            $raw = $_POST['awp_senders_enabled'] ?? [];
            $clean = [
                'email' => ! empty( $raw['email'] ) ? 1 : 0,
                'wa'    => ! empty( $raw['wa'] ) ? 1 : 0,
                'meta'  => ! empty( $raw['meta'] ) ? 1 : 0,
                'block' => ! empty( $raw['block'] ) ? 1 : 0,
            ];
            update_option( 'awp_senders_enabled', $clean );
        
            wp_send_json_success( [ 'message' => __( 'Sender settings saved. Page will now reload.', 'awp' ) ] );
        }

        public function handle_meta_sender_forms() {
            if (!current_user_can('manage_options')) { return; }
            if (isset($_POST['awp_mws_save_settings_nonce']) && wp_verify_nonce($_POST['awp_mws_save_settings_nonce'], 'awp_mws_save_settings_action')) {
                $this->handle_save_settings();
                if (isset($_POST['set_webhook'])) {
                    $this->handle_set_webhook();
                }
                
                if (isset($_POST['fetch_templates'])) {
                    $this->handle_fetch_templates();
                }
            }
        }

        public function get_setting($name) {
            global $wpdb;
            return $wpdb->get_var($wpdb->prepare("SELECT setting_value FROM " . AWP_MWS_SETTINGS_TABLE . " WHERE setting_name = %s", $name));
        }

        public function save_setting($name, $value) {
            global $wpdb;
            $wpdb->replace(AWP_MWS_SETTINGS_TABLE, ['setting_name' => $name, 'setting_value' => $value]);
        }

        public function handle_save_settings() {
            $settings = ['awp_mws_api_token', 'awp_mws_waba_id', 'awp_mws_phone_number_id', 'awp_mws_verify_token', 'awp_mws_app_id', 'awp_mws_app_secret'];
            foreach ($settings as $setting_name) {
                if (isset($_POST[$setting_name])) { $this->save_setting($setting_name, sanitize_text_field(trim($_POST[$setting_name]))); }
            }
            if (isset($_POST['save_settings'])) {
                add_action('admin_notices', function() { echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__('Settings saved.', 'awp') . '</p></div>'; });
            }
        }

        public function handle_fetch_templates() {
            $api_token = trim($this->get_setting('awp_mws_api_token'));
            $waba_id = trim($this->get_setting('awp_mws_waba_id'));
            if (empty($api_token) || empty($waba_id)) { add_action('admin_notices', function() { echo '<div class="notice notice-error is-dismissible"><p>' . esc_html__('Cannot fetch templates. API Token and WhatsApp Business Account ID must be set.', 'awp') . '</p></div>'; }); return; }
            $api_url = "https://graph.facebook.com/v20.0/{$waba_id}/message_templates?fields=name,language&limit=100";
            $response = wp_remote_get($api_url, ['headers' => ['Authorization' => 'Bearer ' . $api_token], 'timeout' => 30]);
            if (is_wp_error($response)) { add_action('admin_notices', function() use ($response) { echo '<div class="notice notice-error is-dismissible"><p>' . esc_html__('Error fetching templates: ', 'awp') . esc_html($response->get_error_message()) . '</p></div>'; }); return; }
            $body = wp_remote_retrieve_body($response);
            $data = json_decode($body, true);
            if (isset($data['data']) && !empty($data['data'])) {
                global $wpdb;
                $wpdb->query("TRUNCATE TABLE " . AWP_MWS_TEMPLATES_TABLE);
                foreach ($data['data'] as $template) { $wpdb->replace(AWP_MWS_TEMPLATES_TABLE, ['template_name' => $template['name'], 'language_code' => $template['language']]); }
                add_action('admin_notices', function() use ($data) { echo '<div class="notice notice-success is-dismissible"><p>' . sprintf(esc_html__('Successfully fetched and stored %d template versions.', 'awp'), count($data['data'])) . '</p></div>'; });
            } else {
                $error_message = isset($data['error']['message']) ? $data['error']['message'] : __('No templates found or an API error occurred.', 'awp');
                add_action('admin_notices', function() use ($error_message) { echo '<div class="notice notice-warning is-dismissible"><p>' . esc_html__('Could not fetch templates: ', 'awp') . esc_html($error_message) . '</p></div>'; });
            }
        }

        public function handle_set_webhook() {
            $verify_token = trim($this->get_setting('awp_mws_verify_token'));
            if (empty($verify_token)) {
                $verify_token = wp_generate_password(32, false);
                $this->save_setting('awp_mws_verify_token', $verify_token);
            }

            $app_id = trim($this->get_setting('awp_mws_app_id'));
            $app_secret = trim($this->get_setting('awp_mws_app_secret'));
            $webhook_url = rest_url('awp/v1/webhook');
            if (empty($app_id) || empty($app_secret)) { add_action('admin_notices', function() { echo '<div class="notice notice-error is-dismissible"><p>' . esc_html__('App ID and App Secret must be set to configure the webhook.', 'awp') . '</p></div>'; }); return; }
            $token_url = "https://graph.facebook.com/oauth/access_token?client_id={$app_id}&client_secret={$app_secret}&grant_type=client_credentials";
            $token_response = wp_remote_get($token_url);
            if (is_wp_error($token_response)) { add_action('admin_notices', function() use ($token_response) { echo '<div class="notice notice-error is-dismissible"><p>' . esc_html__('Failed to get app access token: ', 'awp') . esc_html($token_response->get_error_message()) . '</p></div>'; }); return; }
            $token_body = json_decode(wp_remote_retrieve_body($token_response), true);
            if (!isset($token_body['access_token'])) { add_action('admin_notices', function() { echo '<div class="notice notice-error is-dismissible"><p>' . esc_html__('Could not retrieve a valid app access token from the response.', 'awp') . '</p></div>'; }); return; }
            $app_access_token = $token_body['access_token'];
            $subscribe_url = "https://graph.facebook.com/{$app_id}/subscriptions";
            $subscribe_payload = ['object' => 'whatsapp_business_account', 'callback_url' => $webhook_url, 'verify_token' => $verify_token, 'fields' => 'messages', 'access_token' => $app_access_token];
            $subscribe_response = wp_remote_post($subscribe_url, ['body' => $subscribe_payload]);
            if (is_wp_error($subscribe_response)) { add_action('admin_notices', function() use ($subscribe_response) { echo '<div class="notice notice-error is-dismissible"><p>' . esc_html__('Webhook subscription failed: ', 'awp') . esc_html($subscribe_response->get_error_message()) . '</p></div>'; });
            } else {
                $subscribe_body = json_decode(wp_remote_retrieve_body($subscribe_response), true);
                if (isset($subscribe_body['success']) && $subscribe_body['success']) { add_action('admin_notices', function() { echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__('Webhook successfully subscribed!', 'awp') . '</p></div>'; });
                } else {
                    $error_message = isset($subscribe_body['error']['message']) ? $subscribe_body['error']['message'] : __('An unknown error occurred.', 'awp');
                    add_action('admin_notices', function() use ($error_message) { echo '<div class="notice notice-error is-dismissible"><p>' . esc_html__('Webhook subscription failed: ', 'awp') . esc_html($error_message) . '</p></div>'; });
                }
            }
        }

        private function maybe_save_block_list() {
        
            if ( ! isset( $_POST['awp_block_list_submit'] ) ) {
                return;
            }
            check_admin_referer( 'awp_block_list_nonce', 'awp_block_list_nonce_field' );
        
            $raw  = wp_unslash( $_POST['awp_block_list'] ?? '' );
            $list = array_filter( array_map( 'trim', explode( "\n", $raw ) ) );
        
            ( new AWP_Database_Manager() )->upsert_blocked_numbers( $list );
        
            echo '<div class="notice notice-success is-dismissible"><p>'
                . esc_html__( 'Block list updated successfully.', 'awp' )
                . '</p></div>';
        }
        
        private function get_smtp_settings() : array {
            return array_merge( [
                'enabled'    => 0,
                'host'       => '',
                'port'       => 587,
                'encryption'  => 'tls',  
                'auth'       => 0,
                'user'       => '',
                'pass'       => '',
                'from_email' => get_bloginfo( 'admin_email' ),
                'from_name'  => get_bloginfo( 'name' ),
            ], (array) get_option( 'awp_smtp_settings', [] ) );
        }
        
        public function smtp_send_test_email() {
        
            $this->check_permissions_and_nonce();
        
            $to = sanitize_email( $_POST['to'] ?? '' );
            if ( ! is_email( $to ) ) {
                wp_send_json_error( __( 'Please enter a valid email address.', 'awp' ) );
            }
            $sent = wp_mail(
                $to,
                'Wawp SMTP test' . wp_specialchars_decode( get_bloginfo( 'name' ) ),
                "Wawp SMTP is working! \n\nTime: " . current_time( 'mysql' )
            );
        
            if ( $sent ) {
                wp_send_json_success( __( 'Test message sent – please check the inbox.', 'awp' ) );
            }
            wp_send_json_error( __( 'wp_mail() returned false. Check SMTP settings or server logs.', 'awp' ) );
        }
        
        public function smtp_test_connection() {
            $this->check_permissions_and_nonce();
        
            $o    = $this->get_smtp_settings();
            $host = $o['host'];
            $port = (int) $o['port'];
            $auth = (bool) $o['auth'];
            $enc  = $o['encryption']; 
            $user = $o['user'];
            $pass = $o['pass'];
        
            if ( ! class_exists('\PHPMailer\PHPMailer\PHPMailer', false) ) {
                if ( file_exists( ABSPATH . WPINC . '/PHPMailer/PHPMailer.php' ) ) {
                    require_once ABSPATH . WPINC . '/PHPMailer/PHPMailer.php';
                    require_once ABSPATH . WPINC . '/PHPMailer/SMTP.php';
                    require_once ABSPATH . WPINC . '/PHPMailer/Exception.php';
                } elseif ( file_exists( ABSPATH . WPINC . '/class-phpmailer.php' ) ) {
                    require_once ABSPATH . WPINC . '/class-phpmailer.php';
                    require_once ABSPATH . WPINC . '/class-smtp.php';
                }
            }
        
            try {
                $mail = new \PHPMailer\PHPMailer\PHPMailer(true);
                $mail->isSMTP();
                $mail->Host       = $host;
                $mail->Port       = $port;
                $mail->SMTPAuth   = $auth;
                $mail->SMTPAutoTLS = false;
                if ( $enc !== 'none' ) { $mail->SMTPSecure = $enc; }
                if ( $auth ) { $mail->Username = $user; $mail->Password = $pass; }
    
                $ok = $mail->smtpConnect();
                $mail->smtpClose();
    
                if ( $ok ) { wp_send_json_success( __( 'Connection successful! 🥳', 'awp' ) ); }
                wp_send_json_error( __( 'Could not connect – server rejected the handshake.', 'awp' ) );
            } catch ( \Throwable $e ) {
                wp_send_json_error( sprintf( __( 'Connection failed: %s', 'awp' ), $e->getMessage() ) );
            }
        }


        
        public function render_smtp_settings_form() {
        
            $opts = $this->get_smtp_settings();
            $chk  = function ( $v ) { return $v ? 'checked' : ''; };
        
            ?>
            <form method="post" class="awp-smtp-form" action="">
                <?php wp_nonce_field( 'awp_save_smtp', 'awp_save_smtp_nonce' ); ?>
    
                <div class="awp-card awp-accordion" id="awp-card-smtp">
                    <div class="card-header_row awp-accordion-header">
                        
                        <div class="card-header" style="flex:1;">
                            <h4 class="card-title">
                                <i class="ri-mail-settings-line"></i>
                                <?php esc_html_e( 'SMTP Settings', 'awp' ); ?>
                            </h4>
                            <p><?php esc_html_e(
                                'If enabled, all wp_mail() calls will be routed through this SMTP server.',
                                'awp'
                            ); ?></p>
                            
                        </div>
                        <p class="submit awp_save" style="margin: 0 !important;width: fit-content;">
                            <button type="submit" name="awp_save_smtp" class="awp-btn primary" style="height: 39px;padding: .5rem 1rem;">
                                <?php esc_html_e( 'Save SMTP Settings', 'awp' ); ?>
                            </button>
                        </p>
                    </div>
    
                    <div class="awp-accordion-content">
                        <table class="form-table"><tbody>
                            <tr>
                                <th scope="row"><?php esc_html_e( 'Route emails via SMTP', 'awp' ); ?></th>
                                <td>
                                    <label class="awp-switch">
                                        <input type="checkbox" id="awp-smtp-enabled"
                                               name="awp_smtp[enabled]" value="1" <?php echo $chk( $opts['enabled'] ); ?> >
                                        <span class="custom-control-label"></span>
                                    </label>
                                </td>
                            </tr>
                        </tbody></table>
                        <div id="awp-smtp-settings-fields">
    
                        <table class="form-table"><tbody>
                            <tr>
                                <th><?php esc_html_e( 'SMTP Host', 'awp' ); ?></th>
                                <td>
                                    <input type="text"   name="awp_smtp[host]"
                                           value="<?php echo esc_attr( $opts['host'] ); ?>"
                                           placeholder="smtp.example.com">
                                </td>
                            </tr>
                            <tr>
                                <th><?php esc_html_e( 'Port', 'awp' ); ?></th>
                                <td>
                                    <input type="number" name="awp_smtp[port]"
                                           value="<?php echo (int) $opts['port']; ?>">
                                </td>
                            </tr>
    
                            <tr>
                                <th><?php esc_html_e( 'Encryption', 'awp' ); ?></th>
                                <td>
                                    <select name="awp_smtp[encryption]">
                                        <?php
                                        $encs = [
                                            'none' => __( 'None', 'awp' ),
                                            'ssl'  => 'SSL',
                                            'tls'  => 'TLS'
                                        ];
                                        foreach ( $encs as $val => $label ) {
                                            printf(
                                                '<option value="%s"%s>%s</option>',
                                                esc_attr( $val ),
                                                selected( $opts['encryption'], $val, false ),
                                                esc_html( $label )
                                            );
                                        }
                                        ?>
                                    </select>
                                </td>
                            </tr>
                                <tr>
                                <th><?php esc_html_e( 'From Email', 'awp' ); ?></th>
                                <td>
                                    <input type="email" name="awp_smtp[from_email]"
                                           value="<?php echo esc_attr( $opts['from_email'] ); ?>">
                                </td>
                            </tr>
    
                            <tr>
                                <th><?php esc_html_e( 'Name', 'awp' ); ?></th>
                                <td>
                                    <input type="text"  name="awp_smtp[from_name]"
                                           value="<?php echo esc_attr( $opts['from_name'] ); ?>">
                                </td>
                            </tr>
                                <tr>
                                <th><?php esc_html_e( 'SMTP Authentication', 'awp' ); ?></th>
                                <td>
                                    <label class="awp-switch">
                                        <input type="checkbox" id="awp-smtp-auth"
                                               name="awp_smtp[auth]" value="1" <?php echo $chk( $opts['auth'] ); ?> >
                                        <span class="custom-control-label"></span>
                                    </label>
                                </td>
                            </tr>
                            <tr class="awp-smtp-auth-fields">
                                <th><?php esc_html_e( 'Username', 'awp' ); ?></th>
                                <td>
                                    <input type="text"      name="awp_smtp[user]"
                                           value="<?php echo esc_attr( $opts['user'] ); ?>">
                                </td>
                            </tr>
    
                            <tr class="awp-smtp-auth-fields">
                                <th><?php esc_html_e( 'Password', 'awp' ); ?></th>
                                <td>
                                    <input type="password" name="awp_smtp[pass]"
                                           value="<?php echo esc_attr( $opts['pass'] ); ?>"
                                           autocomplete="new-password">
                                </td>
                            </tr>
    
                        </tbody></table>
    
                        </div>
                    </div>
                </div>
                <div class="awp-card">
                    <div class="card-header_row">
                        <div class="card-header">
                            <h4 class="card-title">
                                <i class="ri-mail-send-line"></i>
                                <?php esc_html_e( 'Send Test Email', 'awp' ); ?>
                            </h4>
                            <p><?php esc_html_e( 'The email address to which you want to send the test.', 'awp' ); ?></p>
                        </div>
                    </div>
                    <p class="btn-group">
                        <input type="email" id="awp-smtp-test-to" placeholder="you@example.com">
                        <button class="awp-btn secondary" id="awp-smtp-test-btn" style="height: 39px;">
                            <?php esc_html_e( 'Send Test', 'awp' ); ?>
                        </button>
                        <div class="card-header">
                            <button class="awp-btn" id="awp-smtp-test-conn" style="height: 40px;">
                                <i class="ri-refresh-line"></i>
                                <?php esc_html_e( 'Check Connection', 'awp' ); ?>
                            </button>
                            
                            <span id="awp-smtp-test-status"></span>
                        </div>
                    </p>
                </div>
    
            </form>
            <?php
        }
        
        private function maybe_save_smtp_settings() {
            if (
                isset( $_POST['awp_save_smtp'] ) &&
                check_admin_referer( 'awp_save_smtp', 'awp_save_smtp_nonce' )
            ) {
                $raw = $_POST['awp_smtp'] ?? [];
                $clean = [
                    'enabled'    => empty( $raw['enabled'] ) ? 0 : 1,
                    'host'       => sanitize_text_field( $raw['host'] ?? '' ),
                    'port'       => (int) ( $raw['port'] ?? 587 ),
                    'encryption' => in_array( $raw['encryption'] ?? 'tls', [ 'none','ssl','tls' ], true ) ? $raw['encryption'] : 'tls',
                    'auth'       => empty( $raw['auth'] ) ? 0 : 1,
                    'user'       => sanitize_text_field( $raw['user'] ?? '' ),
                    'pass'       => sanitize_text_field( $raw['pass'] ?? '' ),
                    'from_email' => sanitize_email( $raw['from_email'] ?? '' ),
                    'from_name'  => sanitize_text_field( $raw['from_name'] ?? '' ),
                ];
                update_option( 'awp_smtp_settings', $clean );
                echo '<div class="notice notice-success is-dismissible"><p>'
                    . esc_html__( 'SMTP settings saved.', 'awp' ) . '</p></div>';
            }
        }
        
        public function auto_check_all_instance_status() {
            check_ajax_referer('awp_nonce', 'nonce');
            if (!current_user_can('manage_options')) {
                wp_send_json_error(__('Unauthorized user.', 'awp'));
            }
    
            $instances = $this->get_all_instances();
            if (!$instances) {
                wp_send_json_error(__('No instances found.', 'awp'));
            }
    
            $updated = 0;
            foreach ($instances as $inst) {
                $result = $this->check_and_update_instance_status($inst->instance_id, $inst->access_token);
                if ($result) {
                    $updated++;
                }
            }
    
            wp_send_json_success([
                'updated_count' => $updated,
                'message'       => sprintf(__('%d instances checked & updated.', 'awp'), $updated),
            ]);
        }
        
        public function awp_get_all_instances_ajax() {
            check_ajax_referer('awp_nonce','nonce');
            if(!current_user_can('manage_options')){
                wp_send_json_error(__('Unauthorized user.','awp'));
            }
            $page = isset($_POST['page'])? absint($_POST['page']) : 1;
            $page_size = 20;
            $offset = ($page-1)*$page_size;
            global $wpdb;
            $table = $wpdb->prefix.'awp_instance_data';
            $rows  = $wpdb->get_results($wpdb->prepare(
                "SELECT * FROM $table LIMIT %d,%d",
                $offset, $page_size
            ));
            wp_send_json_success($rows);
        }
    
        public function get_all_instances() {
            static $cache = null;
        
            if (null !== $cache) {
                return $cache;
            }
        
            global $wpdb;
            $cache = $wpdb->get_results("SELECT * FROM {$wpdb->prefix}awp_instance_data");
            return $cache;
        }
        
        public function get_all_instances_cached($ttl = 60) {
            $key  = 'awp_all_instances';
            $data = get_transient($key);
            if (false !== $data) {
                return $data;
            }
        
            $data = $this->get_all_instances();
            set_transient($key, $data, $ttl);
            return $data;
        }
        
        public function get_online_instances() {
            return $this->database_manager->get_online_instances_cached();
        }
        
        public function add_instance() {
            $this->check_permissions_and_nonce();
            global $wpdb;

            $name         = sanitize_text_field($_POST['name'] ?? '');
            $instance_id  = sanitize_text_field($_POST['instance_id'] ?? '');
            $access_token = sanitize_text_field($_POST['access_token'] ?? '');

            if (!$name || !$instance_id || !$access_token) {
                wp_send_json_error(['message' => __('All fields are required.', 'awp')]);
            }

            $inserted = $wpdb->insert(
                "{$wpdb->prefix}awp_instance_data",
                [
                    'name'         => $name,
                    'instance_id'  => $instance_id,
                    'access_token' => $access_token,
                    'status'       => 'checking',
                    'message'      => __('Verifying connection...', 'awp'),
                ]
            );

            if (false === $inserted) {
                wp_send_json_error(['message' => __('Failed to save the instance to the database.', 'awp')]);
            }

            $this->check_and_update_instance_status($instance_id, $access_token);

            wp_send_json_success(['message' => __('Instance added successfully. Status is being updated.', 'awp')]);
        }
    
        public function delete_instance(){
            $this->check_permissions_and_nonce();
            global $wpdb;
            $id=intval($_POST['id']??0);
            if($id<=0){
                wp_send_json_error(['message' => __('Invalid instance ID.','awp')]);
                return;
            }
            $result=$wpdb->delete("{$wpdb->prefix}awp_instance_data", ['id'=>$id]);
            if($result===false){
                wp_send_json_error(['message' => __('Failed to delete instance.','awp')]);
                return;
            }
            wp_send_json_success(['message' => __('Instance deleted successfully.','awp')]);
        }
    
        public function edit_instance(){
            $this->check_permissions_and_nonce();
            global $wpdb;
            $id=intval($_POST['id']??0);
            $name=sanitize_text_field($_POST['name']??'');
            $instance_id=sanitize_text_field($_POST['instance_id']??'');
            $access_token=sanitize_text_field($_POST['access_token']??'');
            if($id<=0||!$name||!$instance_id||!$access_token){
                wp_send_json_error(['message' => __('All fields are required.','awp')]);
                return;
            }
            $result = $wpdb->update("{$wpdb->prefix}awp_instance_data",[
                'name'         => $name,
                'instance_id'  => $instance_id,
                'access_token' => $access_token,
                'status'       => 'checking',
                'message'      => ''
            ], ['id'=>$id]);
            if($result===false){
                wp_send_json_error(['message' => __('Failed to update instance.','awp')]);
                return;
            }
            $this->check_and_update_instance_status($instance_id,$access_token);
            $st = $this->get_instance_status($instance_id);
            $msg= $this->get_instance_message($instance_id);
            wp_send_json_success(['status'=>$st,'message'=>$msg,'text'=>__('Instance updated successfully.','awp')]);
        }
    
        public function update_status() {
            $this->check_permissions_and_nonce();

            $instance_id  = sanitize_text_field( $_POST['instance_id']  ?? '' );
            $access_token = sanitize_text_field( $_POST['access_token'] ?? '' );

            if ( ! $instance_id || ! $access_token ) {
                wp_send_json_error([ 'message' => __( 'Instance ID & Access Token are required.', 'awp' ) ]);
                return;
            }

            $this->check_and_update_instance_status($instance_id, $access_token);
            $new_status = $this->get_instance_status($instance_id);
            $new_message = $this->get_instance_message($instance_id);
            
            if ( is_null($new_status) ) {
                wp_send_json_error([ 'message' => __( 'Failed to retrieve updated status from database.', 'awp' ) ]);
                return;
            }

            wp_send_json_success([
                'status'  => $new_status,
                'message' => $new_message,
            ]);
        }

        public function send_test_message() {
            $this->check_permissions_and_nonce();

            $instance_id  = sanitize_text_field( $_POST['instance_id']  ?? '' );
            $access_token = sanitize_text_field( $_POST['access_token'] ?? '' );

            if ( ! $instance_id || ! $access_token ) {
                wp_send_json_error( [ 'message' => __( 'Instance ID and Access Token are required.', 'awp' ) ] );
            }

            $phone_number = '447441429009';
            $message      = __( 'Wawp Notification work', 'awp' );

            $result = Wawp_Api_Url::send_message(
                $instance_id,
                $access_token,
                $phone_number,
                $message
            );
            
            $user = wp_get_current_user();
            $customer_name = $user ? ($user->display_name ?: 'Admin Test') : 'Admin Test';
            $response_to_log = $result;
            if (isset($result['full_response'])) {
                $response_to_log = $result['full_response'];
            }

            $log_manager = new AWP_Log_Manager();
            $log_manager->log_notification([
                'user_id'          => $user->ID ?? 0,
                'order_id'         => null,
                'customer_name'    => $customer_name,
                'sent_at'          => current_time( 'mysql' ),
                'whatsapp_number'  => $phone_number,
                'message'          => $message,
                'image_attachment' => null,
                'message_type'     => 'Admin Test Message',
                'wawp_status'      => $response_to_log,
                'resend_id'        => null,
                'instance_id'      => $instance_id,
                'access_token'     => $access_token,
            ]);

            if ( $result['status'] === 'success' ) {
                wp_send_json_success( [
                    'message' => __( 'Test message sent successfully!', 'awp' )
                ] );
            }

            $err = $result['message'] ?? __( 'Unknown error', 'awp' );
            wp_send_json_error( [
                'message' => __( 'Failed to send test message: ', 'awp' ) . $err
            ] );
        }
    
        private function insert_test_message_log($body,$status,$resp=''){
            global $wpdb;
            $table_log = $this->database_manager->get_log_table_name();
            $user_id   = get_current_user_id() ?: 0;
            $number    = $body['number']??'N/A';
            $msg       = $body['message']??'N/A';
            $wpdb->insert($table_log,[
                'user_id'          => $user_id,
                'order_id'         => 0,
                'customer_name'    => (!empty($body['instance_id'])?$body['instance_id']:'N/A'),
                'sent_at'          => current_time('mysql'),
                'whatsapp_number'=> $number,
                'message'          => $msg,
                'image_attachment'=>'',
                'message_type'     => 'test_message',
                'wawp_status'      => $resp,
                'resend_id'        => null
            ]);
        }
    
        private function check_and_update_instance_status( $iid, $atk ) {

            if ( ! $iid || ! $atk ) {
                return false;
            }

            $url  = sprintf(
                '%s/wp-json/awp/v1/session/info?instance_id=%s&access_token=%s',
                $this->wawp_domain,
                rawurlencode( $iid ),
                rawurlencode( $atk )
            );
            $resp = wp_remote_get( $url, [ 'timeout' => 15 ] );

            if ( is_wp_error( $resp ) || wp_remote_retrieve_response_code( $resp ) !== 200 ) {
                $msg = is_wp_error( $resp )
                    ? $resp->get_error_message()
                    : sprintf( 'HTTP%s returned', wp_remote_retrieve_response_code( $resp ) );
                return $this->set_instance_state( $iid, 'offline', $msg );
            }

            $data = json_decode( wp_remote_retrieve_body( $resp ), true );
            if ( json_last_error() !== JSON_ERROR_NONE ) {
                return $this->set_instance_state(
                    $iid, 'offline',
                    'Invalid JSON from /session/info'
                );
            }

            $raw = strtoupper( $data['status'] ?? '' );
            $map = [
                'WORKING'      => 'online',
                'ONLINE'       => 'online',
                'READY'        => 'ready',
                'PAUSED'       => 'paused',
                'OFFLINE'      => 'offline',
                'SCAN_QR_CODE' => 'scan_qr_code',
                'SCAN QR'      => 'scan qr',
                'STOPPED'      => 'stopped',
                'FAILED'       => 'failed'
            ];
            $new_status = $map[ $raw ] ?? 'checking';

            $display = $data['metadata']['display_name']
                             ?? $data['display_name']
                             ?? '';
            $new_msg = ( $new_status === 'online' && $display )
                ? sprintf( __( 'Connected as %s', 'awp' ), sanitize_text_field( $display ) )
                : $raw;

            return $this->set_instance_state( $iid, $new_status, $new_msg );
        }

        private function set_instance_state( $iid, $status, $message ) {
            global $wpdb;
            $table  = "{$wpdb->prefix}awp_instance_data";
            $wpdb->update( $table,
                [ 'status'=>$status, 'message'=>$message ],
                [ 'instance_id'=>$iid ]
            );
            return true;
        }

        private function maybe_autoselect_default_sender( array $online_instances,
            array &$otp_settings,
            string &$curr_login_val,
            string &$curr_signup_val,
            string &$curr_checkout_val,
            string &$curr_resend_val,
            array  &$selected_mult,
            string &$curr_abandoned_cart_val,
            bool   $otp_global,
            bool   $otp_login,
            bool   $otp_signup,
            bool   $otp_checkout,
            bool   $notifications_enabled,
            bool   $abandoned_enabled
        ) {
            if ( empty( $online_instances ) ) {
                return;
            }
            $first = $online_instances[0];

            if ( $otp_global && $otp_login && $curr_login_val === '' ) {
                $curr_login_val = $first->instance_id;
                $otp_settings['instance'] = $curr_login_val;
                update_option( 'awp_otp_settings', $otp_settings );
            }

            if ( $otp_global && $otp_signup && $curr_signup_val === '' && class_exists( 'AWP_Database_Manager' ) ) {
                $dbm = new AWP_Database_Manager();
                $dbm->update_signup_settings( [ 'selected_instance' => (int) $first->id ] );
                $curr_signup_val = (string) $first->id;
            }

            if ( $otp_global && $otp_checkout && $curr_checkout_val === '' ) {
                $curr_checkout_val = $first->instance_id;
                update_option( 'awp_selected_instance', $curr_checkout_val );
            }

            if ( $curr_resend_val === '' ) {
                $curr_resend_val = $first->instance_id;
                update_option( 'awp_selected_log_manager_instance', $curr_resend_val );
            }
            
            if ( $abandoned_enabled && $curr_abandoned_cart_val === '' ) {
                $curr_abandoned_cart_val = $first->id;
                update_option('wawp_abandoned_cart_instance', $curr_abandoned_cart_val);
            }

            if ( $notifications_enabled && empty( $selected_mult ) && class_exists( 'AWP_Database_Manager' ) ) {
                $dbm = $dbm ?? new AWP_Database_Manager();
                $dbm->set_notif_global( (string) $first->id );
                $selected_mult = [ (int) $first->id ];
            }
        }

    
        private function get_instance_status($iid){
            global $wpdb;
            return $wpdb->get_var($wpdb->prepare("SELECT status FROM {$wpdb->prefix}awp_instance_data WHERE instance_id=%s",$iid));
        }
    
        private function get_instance_message($iid){
            global $wpdb;
            return $wpdb->get_var($wpdb->prepare("SELECT message FROM {$wpdb->prefix}awp_instance_data WHERE instance_id=%s",$iid));
        }
    
        public function admin_notices(){
            if(isset($_GET['awp_notice'])){
                $notice=sanitize_text_field($_GET['awp_notice']);
                $type=sanitize_text_field($_GET['awp_notice_type']??'success');
                echo'<div class="notice notice-'.esc_attr($type).' is-dismissible awp-admin-notice"><p>'.esc_html($notice).'</p></div>';
            }
        }
        
        public static function get_notif_sender_instance_ids(): array {
            if (!class_exists('AWP_Database_Manager')) {
                require_once AWP_PLUGIN_DIR . 'includes/class-database-manager.php';
            }
            $dbm = new AWP_Database_Manager();
            $settings = $dbm->get_notif_global();
            $csv = $settings['selected_instance_ids'] ?? '';
            return array_filter(array_map('intval', explode(',', $csv)));
        }
    
        private function check_permissions_and_nonce(){
            if(!current_user_can('manage_options')){
                wp_send_json_error(['message' => __('Unauthorized user.','awp')]);
            }
            check_ajax_referer('awp_nonce','nonce');
        }
    
        private function call_wawp($endpoint,$method='GET',$body=[],$hdrs=[]){
            $url=$this->wawp_domain.$endpoint;
            $args=['timeout'=>5,'headers'=>array_merge(['x-api-key'=>'some-secret-string'],$hdrs)];
            if($method==='POST'){
                $args['method']='POST';
                $args['body']=$body;
                $res=wp_remote_post($url,$args);
            } else {
                if(!empty($body)){
                    $url=add_query_arg($body,$url);
                }
                $res=wp_remote_get($url,$args);
            }
            if(is_wp_error($res)){
                return null;
            }
            $code=wp_remote_retrieve_response_code($res);
            $rb=wp_remote_retrieve_body($res);
            if(in_array($code,[200,201,401,403],true)){
                return['code'=>$code,'body'=>$rb];
            }
            return null;
        }
        
        public function render_otp_senders_form() {
        
            if ( ! current_user_can( 'manage_options' ) ) {
                return;
            }
    
            $otp_global   = $this->get_toggle( 'awp_wawp_otp_enabled' );
            $otp_login    = $this->get_toggle( 'awp_otp_login_enabled' );
            $otp_signup   = $this->get_toggle( 'awp_signup_enabled' );
            $otp_checkout = $this->get_toggle( 'awp_checkout_otp_enabled' );
            $notifications_enabled = $this->get_toggle( 'awp_notifications_enabled' );
            $abandoned_enabled = $this->get_toggle( 'awp_abandoned_carts_enabled' );
    
            if (
                isset( $_POST['awp_choose_sender_submit'] ) &&
                check_admin_referer( 'awp_choose_sender_save', 'awp_choose_sender_nonce' )
            ) {
                $login_val    = sanitize_text_field( $_POST['awp_login_instance']    ?? '' );
                $signup_val   = sanitize_text_field( $_POST['awp_signup_instance']   ?? '' );
                $checkout_val = sanitize_text_field( $_POST['awp_checkout_instance'] ?? '' );
                $resend_val   = sanitize_text_field( $_POST['awp_resend_instance']   ?? '' );
                $abandoned_cart_val = sanitize_text_field($_POST['wawp_abandoned_cart_instance'] ?? '');
    
                $otp_settings             = get_option( 'awp_otp_settings', [] );
                $otp_settings['instance'] = $login_val;
                update_option( 'awp_otp_settings', $otp_settings );
    
                if ( class_exists( 'AWP_Database_Manager' ) ) {
                    $dbm = new AWP_Database_Manager();
                    $dbm->update_signup_settings( [ 'selected_instance' => (int) $signup_val ] );
                }
    
                update_option( 'awp_selected_instance',                 $checkout_val );
                update_option( 'awp_selected_log_manager_instance', $resend_val );
                if ( $abandoned_enabled ) {
                    update_option( 'wawp_abandoned_cart_instance', $abandoned_cart_val );
                } else {
                    update_option( 'wawp_abandoned_cart_instance', '' );
                }
    
                if ( $notifications_enabled && class_exists( 'AWP_Database_Manager' ) ) {
                 $dbm  = $dbm ?? new AWP_Database_Manager();
                 $sel  = isset( $_POST['wawp_notif_selected_instance'] )
                    ? array_unique( array_map( 'intval', (array) $_POST['wawp_notif_selected_instance'] ) )
                    : [];
                 $dbm->set_notif_global( implode( ',', $sel ) );
             }
                
                echo '<div class="notice notice-success is-dismissible"><p>';
                esc_html_e( 'Settings saved successfully!', 'awp' );
                echo '</p></div>';
            }
    
            $otp_settings    = get_option( 'awp_otp_settings', [] );
            $curr_login_val  = $otp_settings['instance'] ?? '';
    
            $curr_signup_val  = '';
            if ( class_exists( 'AWP_Database_Manager' ) ) {
                $dbm      = $dbm ?? new AWP_Database_Manager();
                $signup_row = $dbm->get_signup_settings();
                if ( ! empty( $signup_row['selected_instance'] ) ) {
                    $curr_signup_val = (string) $signup_row['selected_instance'];
                }
            }
    
            $curr_checkout_val  = get_option( 'awp_selected_instance',               '' );
            $curr_resend_val    = get_option( 'awp_selected_log_manager_instance', '' );
            $curr_abandoned_cart_val = get_option('wawp_abandoned_cart_instance', '');
    
            if ( class_exists( 'AWP_Database_Manager' ) ) {
                $notif_settings = $dbm->get_notif_global();
                $selected_csv   = $notif_settings['selected_instance_ids'] ?? '';
                $selected_mult  = array_filter( array_map( 'intval', explode( ',', $selected_csv ) ) );
            } else {
                $selected_mult = [];
            }
    
            $online_instances = class_exists( 'AWP_Database_Manager' ) ? ( new AWP_Database_Manager() )->get_online_instances_cached() : [];
            
            $this->maybe_autoselect_default_sender(
             $online_instances,
             $otp_settings,
             $curr_login_val,
             $curr_signup_val,
             $curr_checkout_val,
             $curr_resend_val,
             $selected_mult,
             $curr_abandoned_cart_val,
             $otp_global,
             $otp_login,
             $otp_signup,
             $otp_checkout,
             $notifications_enabled,
             $abandoned_enabled
         );
    
            $otp_cols = [];
            if ( $otp_global && $otp_login ) {
                $otp_cols[] = [
                    'type'  => 'instance_id',
                    'label' => __( 'Login via WhatsApp', 'awp' ),
                    'name'  => 'awp_login_instance',
                    'value' => $curr_login_val,
                    'description' => 'Send OTP codes via WhatsApp to customers.',
                ];
            }
            if ( $otp_global && $otp_signup ) {
                $otp_cols[] = [
                    'type'  => 'numeric_id',
                    'label' => __( 'Confirm New Users', 'awp' ),
                    'name'  => 'awp_signup_instance',
                    'value' => $curr_signup_val,
                    'description' => 'Send OTP codes via WhatsApp to verify customers before creating an account.',
                ];
            }
            if ( $otp_global && $otp_checkout ) {
                $otp_cols[] = [
                    'type'  => 'instance_id',
                    'label' => __( 'Verify WooCommerce Orders', 'awp' ),
                    'name'  => 'awp_checkout_instance',
                    'value' => $curr_checkout_val,
                    'description' => 'Send OTP codes via WhatsApp to confirm customer phone number before completing checkout.',
                ];
            }
    
    
            $resend_col = [
                'type'  => 'instance_id',
                'label' => __( 'Resend Failed Messages', 'awp' ),
                'name'  => 'awp_resend_instance',
                'value' => $curr_resend_val,
                'description' => 'Resends messages that failed to send from any of the other features.',
            ];

            $abandoned_cart_col = [
                'type' => 'numeric_id',
                'label' => __('Wawp Abandoned Cart', 'awp'),
                'name' => 'wawp_abandoned_cart_instance',
                'value' => $curr_abandoned_cart_val,
                'description' => 'Choose the instance to send abandoned cart recovery messages.',
            ];
    
            $any_otp = ! empty( $otp_cols );
            $has_any = ( $any_otp || true );
            ?>
            <form method="post" action="">
                <div class="awp-card awp-accordion" id="awp-card-sender" style="margin:20px 0;">
    
                    <div class="card-header_row awp-accordion-header">
                        <div class="card-header">
                            <h4 class="card-title"><?php esc_html_e( 'Choose WhatsApp Sender', 'awp' ); ?></h4>
                            <p><?php esc_html_e( 'Select the WhatsApp number that will be used to send messages for each feature.', 'awp' ); ?></p>
                        </div>
                    </div>
    
                    <div class="awp-accordion-content">
                        <?php wp_nonce_field( 'awp_choose_sender_save', 'awp_choose_sender_nonce' ); ?>
    
                        <div class="instances-setup">
                            <?php if ( ! $has_any ) : ?>
    
                                <p style="color:#a00;"><?php esc_html_e( 'All features are disabled, so there are no sender fields to configure.', 'awp' ); ?></p>
    
                            <?php else : ?>
                                <?php if ( $notifications_enabled ) : ?>
                                <div class="instance-select" style="min-width:100%;">
                                    <label for="wawp_notif_selected_instance">
                                        <?php esc_html_e( 'WhatsApp Notifications', 'awp' ); ?>
                                    </label>
                                    <?php if ( ! empty( $online_instances ) ) : ?>
                                        <p><?php esc_html_e( 'Choose one or more numbers to send notification events for users and admins.', 'awp' ); ?></p>
                                        <select name="wawp_notif_selected_instance[]" id="wawp_notif_selected_instance"
                                                multiple="multiple" class="wawp-instance-multi" style="width:100%;">
                                            <?php foreach ( $online_instances as $inst ) : ?>
                                                <option value="<?php echo (int) $inst->id; ?>"
                                                    <?php selected( in_array( (int) $inst->id, $selected_mult, true ), true ); ?>>
                                                    <?php echo esc_html( "{$inst->name} ({$inst->instance_id})" ); ?>
                                                </option>
                                            <?php endforeach; ?>
                                        </select>
                                    <?php else : ?>
                                        <p style="color:#a00;margin:0;">
                                            <?php esc_html_e( 'No online instances found.', 'awp' ); ?>
                                        </p>
                                    <?php endif; ?>
                                </div>
                                <?php endif; ?>

                                <?php
                                
                                
                                if ( $any_otp ) {
                                    foreach ( $otp_cols as $col ) {
                                        echo $this->render_column( $col, $online_instances );
                                    }
                                }
    
                                echo $this->render_column( $resend_col, $online_instances );
                                if ( $abandoned_enabled ) {
                                    echo $this->render_column( $abandoned_cart_col, $online_instances );
                                }
                                ?>
    
    
                            <?php endif; ?>
    
                            <p class="submit awp_save">
                                <button type="submit" name="awp_choose_sender_submit" class="awp-btn primary"><?php esc_html_e( 'Save Changes', 'awp' ); ?></button>
                            </p>
                        </div>
                    </div>
                </div>
            </form>
            <?php
        }
        
        
    
        private function get_toggle($option){
            $val=get_option($option,'');
            if($val===''){
                return true;
            }
            return (bool)$val;
        }
    
        private function render_column($col,$instances){
            ob_start();
            echo'<div class="instance-select">';
            echo'<label for="'.esc_attr($col['name']).'">'.esc_html($col['label']).'</label>';
            echo'<p>'.esc_html($col['description']).'</p>';
            if(!empty($instances)){
                echo'<select id="'.esc_attr($col['name']).'" name="'.esc_attr($col['name']).'">';
                echo'<option value="">'.esc_html__('-- Select an online instance --','awp').'</option>';
                foreach($instances as $inst){
                    if($col['type']==='numeric_id'){
                        $val=(int)$inst->id;
                        $sel=selected($col['value'],$val,false);
                        echo'<option value="'.esc_attr($val).'" '.$sel.'>'.esc_html($inst->name.' (ID#'.$inst->id.')').'</option>';
                    } else {
                        $iid=$inst->instance_id;
                        $sel=selected($col['value'],$iid,false);
                        echo'<option value="'.esc_attr($iid).'" '.$sel.'>'.esc_html($inst->name.' ('.$inst->instance_id.')').'</option>';
                    }
                }
                echo'</select>';
            } else {
                echo'<p style="color:#a00;margin:0;">'.esc_html__('No online instances found. ','awp').'</p>';
            }
            echo'</div>';
            return ob_get_clean();
        }
        
    }