(function ($) {
    'use strict';

    /* =====================================================
     * Globals
     * ===================================================*/
    const wawpData = window.wawpWizardData || {};
    const awpAjax = window.wawpWizardAjax || {};

    let currentStep = parseInt(wawpData.initialStep || 1, 10);
    const isSysInfoEnabled = wawpData.isSysInfoEnabled;
    const isCountryCodeEnabled = wawpData.isCountryCodeEnabled;
    const isWaSenderEnabled = wawpData.isWaSenderEnabled;
    const isStep3Enabled = isSysInfoEnabled;

    /* =====================================================
     * AJAX and UI Helpers
     * ===================================================*/
    function displayAdminNotice(msg, type = 'success') {
        const noticeContainer = $('#wawp-wizard-content');
        let notice = noticeContainer.find('.wawp-wizard-notice');
        if (notice.length === 0) {
            notice = $(`<div class="wawp-wizard-notice"></div>`);
            noticeContainer.prepend(notice);
        }
        
        notice.removeClass('notice-success notice-error').addClass(`notice-${type}`);
        notice.html(`<p>${msg}</p>`).fadeIn();

        if (type !== 'error') {
            setTimeout(() => {
                notice.fadeOut(500, () => notice.remove());
            }, 5000);
        }
    }

    function showSpinner(isSilent = false, container = '#wawp-wizard-content') {
        if (!isSilent) {
            let spinner = $(container).find('.awp-loading-spinner-overlay');
            if(spinner.length === 0) {
                 $(container).append('<div class="awp-loading-spinner-overlay" style="display:none;"><div class="awp-loading-spinner"><i class="ri-loader-4-line"></i></div></div>');
                 spinner = $(container).find('.awp-loading-spinner-overlay');
            }
            spinner.fadeIn(200);
        }
    }

    function hideSpinner(isSilent = false, container = '#wawp-wizard-content') {
        if (!isSilent) {
            $(container).find('.awp-loading-spinner-overlay').fadeOut(200);
        }
    }

    function requestAjax(data, onSuccess, isSilent = false, onError = null) {
        showSpinner(isSilent);
        $.post(awpAjax.ajax_url, data)
            .done(res => {
                hideSpinner(isSilent);
                if (res.success) {
                    if (typeof onSuccess === 'function') onSuccess(res);
                } else {
                    const errorMessage = res.data && res.data.message ? res.data.message : (awpAjax.unexpectedError || 'An unexpected error occurred.');
                    if (typeof onError === 'function') {
                        onError(res.data || { message: errorMessage });
                    } else if (!isSilent) {
                        displayAdminNotice(errorMessage, 'error');
                    }
                }
            })
            .fail((jqXHR, textStatus, errorThrown) => {
                hideSpinner(isSilent);
                const errorMsg = `${awpAjax.unexpectedError || 'An unexpected error occurred.'} (Network Error: ${textStatus} ${errorThrown || ''})`;
                if (typeof onError === 'function') {
                    onError({ message: errorMsg, raw_response_body_preview: jqXHR.responseText });
                } else if (!isSilent) {
                    displayAdminNotice(errorMsg, 'error');
                }
            });
    }

    /* =====================================================
     * Wizard Core Logic
     * ===================================================*/
    function updateProgress() {
        $('.wawp-progress-step').removeClass('active');
        for (let i = 1; i <= 4; i++) { // Adjusted total steps
            if (i <= currentStep) {
                $('.wawp-progress-step.step-' + i).addClass('active');
            }
        }
    }

    function updateUrlStep(step) {
        const url = new URL(window.location.href);
        url.searchParams.set('wawp_popup_step', step);
        history.replaceState({}, '', url.toString());
    }

    function removeUrlStep() {
        const url = new URL(window.location.href);
        url.searchParams.delete('wawp_popup_step');
        history.replaceState({}, '', url.toString());
    }

    function showStep(step) {
        $('.wawp-step').removeClass('active');
        $('#wawp-step-' + step).addClass('active');
        currentStep = step;
        if (step <= 4) { // Adjusted total steps
            updateProgress();
        }
        updateUrlStep(step);
        $(document).trigger('wawp-show-step', [step]);
    }

    function getNextStep(current) {
        let next = current + 1;
        if (next === 3 && !isStep3Enabled) next++;
        return next;
    }

    function getPrevStep(current) {
        let prev = current - 1;
        if (prev === 3 && !isStep3Enabled) prev--;
        return prev;
    }

    /* =====================================================
     * Final Checks for Step 5 (was Step 6)
     * ===================================================*/
    function runFinalChecks() {
        const $finalChecksContainer = $('#wawp-final-checks');
        const $finishScreen = $('#wawp-finish-screen');
        const $finishButton = $('#wawp-step-4 .wawp-next-btn'); // Adjusted step number

        if (!isSysInfoEnabled) {
            $finalChecksContainer.hide();
            $finishScreen.show();
            $finishButton.text('Finish').prop('disabled', false);
            return;
        }

        const $checklist = $finalChecksContainer.find('.wawp-checklist');
        const $progressBar = $finalChecksContainer.find('.wawp-progress-bar');
        const $issuesNotice = $('#wawp-final-issues-notice');

        $finalChecksContainer.show();
        $finishScreen.hide();
        $issuesNotice.hide();
        $finishButton.text('Checking...').prop('disabled', true);

        $checklist.find('.icon').removeClass('success error').addClass('loading');
        $progressBar.css('width', '0%');

        const checks = [
            { id: '#check-db', key: 'db_ok' },
            { id: '#check-cron-status', key: 'cron_status_ok' },
            { id: '#check-cron-const', key: 'cron_const_ok' },
            { id: '#check-instances', key: 'instances_ok' },
            { id: '#check-issues', key: 'total_issues' }
        ];
        let completedChecks = 0;
        
        const updateCheckStatus = (selector, isOk) => {
            const $item = $(selector);
            setTimeout(() => {
                $item.find('.icon').removeClass('loading').addClass(isOk ? 'success' : 'error');
                completedChecks++;
                const percentage = (completedChecks / checks.length) * 100;
                $progressBar.css('width', percentage + '%');

                if (completedChecks === checks.length) {
                    $finishButton.prop('disabled', false);
                }
            }, completedChecks * 400);
        };

        requestAjax({ action: 'wawp_wizard_get_final_checks', nonce: awpAjax.nonce }, res => {
            if (res.success) {
                const data = res.data;
                updateCheckStatus('#check-db', data.db_ok);
                updateCheckStatus('#check-cron-status', data.cron_status_ok);
                updateCheckStatus('#check-cron-const', data.cron_const_ok);
                updateCheckStatus('#check-instances', data.instances_ok);
                updateCheckStatus('#check-issues', data.total_issues === 0);

                setTimeout(() => {
                    if (data.total_issues > 0) {
                        $issuesNotice.show();
                    } else {
                        $finalChecksContainer.hide();
                        $finishScreen.show();
                    }
                    $finishButton.text('Finish');
                }, (checks.length + 1) * 400);
            }
        });
    }

    /* =====================================================
     * Event Handlers
     * ===================================================*/
    $(document).ready(function () {
        const urlStep = new URL(window.location.href).searchParams.get('wawp_popup_step');
        if (urlStep) {
            $('#wawp-wizard-modal').fadeIn();
            showStep(parseInt(urlStep, 10));
        }

        $('#wawp-floating-launch').on('click', function () {
            $('#wawp-wizard-modal').fadeIn();
            showStep(currentStep);
        });

        $('#wawp-wizard-close').on('click', function (e) {
            e.preventDefault();
            $('#wawp-wizard-modal').fadeOut();
            removeUrlStep();
        });

        $('.wawp-next-btn').on('click', function (e) {
            e.preventDefault();
            const $this = $(this);

            if ($this.is('#wawp-finish-button') || ($this.closest('.wawp-step').is('#wawp-step-4'))) { // Adjusted step number
                $('#wawp-wizard-modal').fadeOut();
                removeUrlStep();
                return;
            }

            let nextStep = getNextStep(currentStep);
            showStep(nextStep);
        });

        $('.wawp-prev-btn').on('click', function (e) {
            e.preventDefault();
            let prevStep = getPrevStep(currentStep);
            showStep(prevStep);
        });

        $(document).on('wawp-show-step', function (e, stepNo) {
            if (stepNo === 4) { // This is now the final checks step
                runFinalChecks();
            }
        });
        // Re-run checks button
        $(document).on('click', '#wawp-rerun-checks-btn, #wawp-recheck-status-btn', function(e) {
            e.preventDefault();
            runFinalChecks();
        });

        // Sync/Repair logs accordion
        $(document).on('click', '.wawp-log-toggle', function (e) {
            e.preventDefault();
            const $acc = $(this).closest('.wawp-accordion');
            const open = $acc.toggleClass('open').hasClass('open');
            $(this).text(open ? 'Hide logs' : 'Show logs');
        });
    });
    
    

    window.updateWawpLogs = function(html) {
        $('#wawp-sync-logs').html(html);
    };

})(jQuery);
