(function () {
  // ===========
  // CSS helpers
  // ===========
  function injectCSS() {
    if (document.getElementById('wawp-flag-alignment-style')) return;
    var style = document.createElement('style');
    style.id = 'wawp-flag-alignment-style';
    style.type = 'text/css';
    style.textContent = [
      /* flag alignment */
      '.iti.iti--flag-right .iti__flag-container{left:auto!important;right:0!important;}',
      '.iti.iti--flag-right input{padding-left:12px!important;padding-right:44px!important;}',
      '.iti:not(.iti--flag-right) .iti__flag-container{left:0!important;right:auto!important;}',
      '.iti:not(.iti--flag-right) input{padding-left:44px!important;padding-right:12px!important;}',
      '.iti.iti--flag-right .iti__country-list{text-align:right!important;}',
      /* status row */
      '.intl-tel-status{display:flex;align-items:center;gap:.5rem;margin-top:5px;font-size:.9em;}',
      '.intl-status-text{white-space:nowrap;}',
      '.intl-status-text.intl-green{color:green;}',
      '.intl-status-text.intl-red{color:red;}',
      '.intl-status-text.intl-blue{color:blue;}',
      '.intl-hidden{display:none!important;}',
      /* carrier label */
      '.intl-carrier{white-space:nowrap;}',
      /* WhatsApp chip */
      '.awp-wa-chip{display:inline-flex;align-items:center;border:1px solid #d1d5db;border-radius:999px;padding:.15rem .55rem;font-size:.85em;line-height:1;gap:.35rem;}',
      '.awp-wa-chip svg{width:14px;height:14px;display:inline-block;}',
      '.awp-wa-chip--found{background:#ecfdf5;border-color:#a7f3d0;color:#065f46;}',
      '.awp-wa-chip--not{background:#fef2f2;border-color:#fecaca;color:#991b1b;}'
    ].join('');
    document.head.appendChild(style);
  }
  injectCSS();

  // ===========
  // Settings / utils
  // ===========
  var settings = (typeof wooIntlTelSettings !== 'undefined') ? wooIntlTelSettings : {};
  if (typeof settings.enableCarrierLookup === 'undefined') settings.enableCarrierLookup = true;
if (typeof settings.enableWhatsAppCheck === 'undefined') settings.enableWhatsAppCheck = true;
  var LEGACY_UTILS_FALLBACK = 'https://cdn.jsdelivr.net/npm/intl-tel-input@17.0.19/build/js/utils.js';

  var utilsLoaded = !!window.intlTelInputUtils;
  var utilsLoading = false;

  function loadUtilsOnce(cb) {
    if (utilsLoaded) { cb && cb(); return; }
    if (utilsLoading) {
      var t = setInterval(function () {
        if (utilsLoaded) { clearInterval(t); cb && cb(); }
      }, 50);
      return;
    }
    utilsLoading = true;

    if (window.intlTelInputUtils) { utilsLoaded = true; cb && cb(); return; }

    var src = (settings.utilsUrl && settings.utilsUrl.indexOf('http') === 0) ? settings.utilsUrl : LEGACY_UTILS_FALLBACK;
    var s = document.createElement('script');
    s.src = src;
    s.async = true;
    s.onload = function () { /* ready */ };
    s.onerror = function () {
      if (src !== LEGACY_UTILS_FALLBACK) {
        var s2 = document.createElement('script');
        s2.src = LEGACY_UTILS_FALLBACK;
        s2.async = true;
        document.head.appendChild(s2);
      }
    };
    document.head.appendChild(s);

    var watch = setInterval(function () {
      if (window.intlTelInputUtils) {
        clearInterval(watch);
        utilsLoaded = true;
        cb && cb();
      }
    }, 50);

    setTimeout(function () {
      clearInterval(watch);
      utilsLoaded = true; // proceed even if blocked
      cb && cb();
    }, 4000);
  }

  function applyAlignmentClass(inputEl) {
    var align = (settings.countryCodeAlignment || 'auto');
    if (align === 'auto') align = settings.isRTL ? 'right' : 'left';
    var wrap = inputEl.closest('.iti');
    if (!wrap) return;
    if (align === 'right') wrap.classList.add('iti--flag-right');
    else wrap.classList.remove('iti--flag-right');
  }

  // ===========
  // Status row helpers
  // ===========
  function ensureStatusBelow(inputEl) {
    var wrap = inputEl.closest('.iti');
    if (!wrap || !wrap.parentNode) return null;

    var row = wrap.parentNode.querySelector('.intl-tel-status');
    if (!row) {
      row = document.createElement('div');
      row.className = 'intl-tel-status';
      // fixed layout: status text + carrier + WA chip
      var status = document.createElement('span');
      status.className = 'intl-status-text';
      var carrier = document.createElement('span');
      carrier.className = 'intl-carrier';
      row.appendChild(status);
      row.appendChild(carrier);
      wrap.parentNode.insertBefore(row, wrap.nextSibling);
    }
    return row;
  }

  function setStatusText(row, text, tone) {
    var el = row.querySelector('.intl-status-text');
    if (!el) return;
    el.textContent = text || '';
    el.classList.remove('intl-green', 'intl-red', 'intl-blue');
    if (tone === 'green') el.classList.add('intl-green');
    else if (tone === 'red') el.classList.add('intl-red');
    else if (tone === 'blue') el.classList.add('intl-blue');
  }

  function showStatusText(row, show) {
    var el = row.querySelector('.intl-status-text');
    if (!el) return;
    if (show) el.classList.remove('intl-hidden'); else el.classList.add('intl-hidden');
  }

  function setCarrier(row, text) {
    var el = row.querySelector('.intl-carrier');
    if (!el) return;
    el.textContent = text || '';
  }

  function clearWhatsAppChip(row) {
    var chip = row.querySelector('.awp-wa-chip');
    if (chip) chip.remove();
    // whenever chip is removed, allow the status text to be visible again
    showStatusText(row, true);
  }

  function setWhatsAppChip(row, state) {
    // state: 'found' | 'not' | 'clear'
    if (state === 'clear') { clearWhatsAppChip(row); return; }

    var chip = row.querySelector('.awp-wa-chip');
    if (!chip) {
      chip = document.createElement('span');
      chip.className = 'awp-wa-chip';
      chip.innerHTML =
        '<svg viewBox="0 0 24 24" aria-hidden="true"><path d="M12.04 2C6.56 2 2.1 6.45 2.1 11.93c0 2.09.56 4 1.54 5.66L2 22l4.56-1.59c1.59.87 3.42 1.37 5.37 1.37 5.48 0 9.93-4.45 9.93-9.93C21.86 6.45 17.52 2 12.04 2zm0 17.86c-1.79 0-3.45-.53-4.84-1.44l-.35-.22-2.7.94.93-2.6-.23-.38a8.06 8.06 0 01-1.38-4.43c0-4.47 3.6-8.07 8.07-8.07s8.07 3.6 8.07 8.07-3.6 8.13-8.07 8.13zm4.6-6.08c-.25-.12-1.46-.72-1.68-.8-.22-.08-.38-.12-.54.12s-.62.8-.77.96c-.14.16-.28.18-.52.06-.25-.12-1.04-.38-1.99-1.2-.73-.65-1.23-1.44-1.37-1.68-.14-.24-.01-.37.11-.49.11-.11.25-.28.37-.43.12-.14.16-.24.25-.41.08-.16.04-.31-.02-.43-.06-.12-.54-1.3-.74-1.79-.2-.48-.4-.41-.54-.41l-.46-.01c-.16 0-.43.06-.66.31-.22.24-.86.84-.86 2.04s.88 2.36 1 2.53c.12.16 1.73 2.64 4.19 3.7.59.25 1.05.4 1.41.5.59.19 1.13.16 1.55.1.47-.07 1.46-.6 1.67-1.19.21-.59.21-1.09.14-1.19-.06-.1-.22-.16-.47-.28z"></path></svg>';
      row.appendChild(chip);
    }
    chip.classList.remove('awp-wa-chip--found', 'awp-wa-chip--not');

    if (state === 'found') {
      chip.classList.add('awp-wa-chip--found');
      chip.textContent = '';
      chip.insertAdjacentHTML('beforeend', '<span>' + ((window.awpTelInputStrings && awpTelInputStrings.waExists) ? awpTelInputStrings.waExists : 'WhatsApp found') + '</span>');
      // hide the "✓ Valid" text when WA status is present
      showStatusText(row, false);
    } else if (state === 'not') {
      chip.classList.add('awp-wa-chip--not');
      chip.textContent = '';
      chip.insertAdjacentHTML('beforeend', '<span>' + ((window.awpTelInputStrings && awpTelInputStrings.waNotFound) ? awpTelInputStrings.waNotFound : 'WhatsApp not found') + '</span>');
      // hide the "✓ Valid" text when WA status is present
      showStatusText(row, false);
    }
  }

  // Write + dispatch (and re-apply shortly to defeat React/Gutenberg clearing)
  function writeValueAndDispatch(el, value) {
    var desc = Object.getOwnPropertyDescriptor(HTMLInputElement.prototype, 'value');
    if (desc && desc.set) { desc.set.call(el, value); } else { el.value = value; }
    el.dispatchEvent(new Event('input',  { bubbles: true }));
    el.dispatchEvent(new Event('change', { bubbles: true }));
    // React/Gutenberg sometimes overwrites on blur → write again shortly
    setTimeout(function () {
      if (desc && desc.set) { desc.set.call(el, value); } else { el.value = value; }
      el.dispatchEvent(new Event('input',  { bubbles: true }));
      el.dispatchEvent(new Event('change', { bubbles: true }));
    }, 30);
  }

  // Normalize to E.164 (digits in the field)
  function normalizeIntoField(inputEl, iti) {
    var raw = (inputEl.value || '').trim();
    if (!raw) return;

    var cd   = iti.getSelectedCountryData ? iti.getSelectedCountryData() : {};
    var dial = cd && cd.dialCode ? cd.dialCode : '';
    var iso2 = cd && cd.iso2 ? cd.iso2 : (settings.defaultCountry || 'us');

    var hasPlus   = raw.charAt(0) === '+';
    var rawDigits = hasPlus ? raw.slice(1).replace(/\D+/g, '') : raw.replace(/\D+/g, '');

    var combinedDigits;
    if (hasPlus) {
      combinedDigits = rawDigits;
    } else {
      var trimmed = rawDigits.replace(/^0+/, '');
      combinedDigits = (dial && !trimmed.startsWith(dial)) ? (dial + trimmed) : trimmed;
    }

    var candidatePlus = '+' + combinedDigits;
    var finalE164 = candidatePlus;

    if (window.intlTelInputUtils && typeof window.intlTelInputUtils.formatNumber === 'function') {
      try {
        finalE164 = window.intlTelInputUtils.formatNumber(
          candidatePlus, iso2, window.intlTelInputUtils.numberFormat.E164
        ) || candidatePlus;
      } catch (_) {}
    } else {
      try {
        var e164 = iti.getNumber(
          window.intlTelInputUtils && window.intlTelInputUtils.numberFormat
            ? window.intlTelInputUtils.numberFormat.E164
            : undefined
        );
        if (e164) finalE164 = e164;
      } catch (_) {}
    }

    var onlyDialPlus = dial ? ('+' + dial) : '+';
    var good = finalE164 && finalE164 !== '+' && finalE164 !== onlyDialPlus && finalE164.replace('+', '').length >= 5;
    var writeThis = good ? finalE164.replace(/^\+/, '') : combinedDigits;

    writeValueAndDispatch(inputEl, writeThis);
  }

  // Error mapping
  function humanizeErrorCode(code) {
    switch (code) {
      case 0: return 'Possible but not valid';
      case 1: return 'Invalid country code';
      case 2: return 'Too short';
      case 3: return 'Too long';
      case 4: return 'Not a number';
      case 5: return 'Invalid length';
      default: return null;
    }
  }

  function maybeCheckWhatsAppExists(inputEl, row, e164, isValidNow) {
      if (!settings.enableWhatsAppCheck) {
      setWhatsAppChip(row, 'clear');
      return;
    }
    try {
      if (!isValidNow) { setWhatsAppChip(row, 'clear'); return; }
      if (!settings.waExistsProxy) return; // proxy not configured

      var digits = (e164 || '').replace(/^\+/, '').replace(/\D/g, '');
      if (digits.length < 6) { setWhatsAppChip(row, 'clear'); return; }

      // Debounce same number
      if (inputEl.dataset.waLastChecked === digits) return;
      inputEl.dataset.waLastChecked = digits;

      var url = settings.waExistsProxy + '?phone=' + encodeURIComponent(digits);

      fetch(url, { credentials: 'omit' })
        .then(function(r){ return r.ok ? r.json() : null; })
        .then(function(data){
          if (!data || typeof data.numberExists === 'undefined') {
            setWhatsAppChip(row, 'clear'); // no result -> keep valid text visible
            return;
          }
          setWhatsAppChip(row, data.numberExists ? 'found' : 'not');
        })
        .catch(function(){ setWhatsAppChip(row, 'clear'); });
    } catch (_) {
      setWhatsAppChip(row, 'clear');
    }
  }

  // Strict validity + carrier + WA existence
  function updateStatus(inputEl, iti) {
    var row = ensureStatusBelow(inputEl);
    if (!row) return;

    var val = (inputEl.value || '').trim();

    if (document.activeElement === inputEl && val !== '') {
      setStatusText(
        row,
        (typeof awpTelInputStrings !== 'undefined' && awpTelInputStrings.waiting) ?
          awpTelInputStrings.waiting : 'Number updates automatically after writing.',
        'blue'
      );
      setCarrier(row, '');
      clearWhatsAppChip(row);
      inputEl.setAttribute('aria-invalid', 'false');
      return;
    }

    if (!val) {
      setStatusText(row, '', '');
      setCarrier(row, '');
      clearWhatsAppChip(row);
      inputEl.setAttribute('aria-invalid', 'false');
      return;
    }

    var valid  = false;
    var e164   = null;
    var errTxt = null;

    try {
      var cd     = (iti.getSelectedCountryData && iti.getSelectedCountryData()) || {};
      var dial   = (cd && cd.dialCode) ? cd.dialCode : '';
      var digits = (val || '').replace(/\D+/g, '');
      var looksInternational = dial && digits.indexOf(dial) === 0;
      var fullDigits = looksInternational ? digits : (dial + digits);
      e164 = '+' + fullDigits;

      if (window.intlTelInputUtils) {
        valid = window.intlTelInputUtils.isValidNumber(e164);
        if (!valid && typeof window.intlTelInputUtils.getValidationError === 'function') {
          var code = window.intlTelInputUtils.getValidationError(e164);
          errTxt = humanizeErrorCode(code);
        }
      } else {
        valid = false; // neutral until utils ready
      }
    } catch (_) {
      valid = false;
    }

    // Carrier fetch helper (always keeps carrier visible)
    function maybeLookupCarrier(tagAsGuess) {
    if (!settings.enableCarrierLookup) return;
      if (!settings.lookupCarrierEndpoint || !e164 || e164[0] !== '+') return;
      if (e164.replace(/\D/g, '').length < 9) return;

      if (inputEl.dataset.carrierE164 === e164 && inputEl.dataset.carrierTagged === (tagAsGuess ? '1' : '0')) return;
      inputEl.dataset.carrierE164 = e164;
      inputEl.dataset.carrierTagged = tagAsGuess ? '1' : '0';

      fetch(settings.lookupCarrierEndpoint + '?e164=' + encodeURIComponent(e164), { credentials: 'omit' })
        .then(function (r) { return r.ok ? r.json() : null; })
        .then(function (data) {
          if (!data || !data.carrier) return;
          var text = data.carrier;
          if (tagAsGuess) text += ' (guess)';
          setCarrier(row, text); // carrier stays even if WA chip shows/hides the status text
        })
        .catch(function(){});
    }

    if (valid) {
      setStatusText(
        row,
        (window.awpTelInputStrings && awpTelInputStrings.valid) ? awpTelInputStrings.valid : '✓ Valid phone number.',
        'green'
      );
      showStatusText(row, true); // show initially; will hide if WA chip appears
      inputEl.setAttribute('aria-invalid', 'false');
      maybeLookupCarrier(false);
      // After carrier: check WhatsApp existence (hides status text when shown)
      maybeCheckWhatsAppExists(inputEl, row, e164, true);
    } else {
      setStatusText(
        row,
        ((window.awpTelInputStrings && awpTelInputStrings.invalid) ? awpTelInputStrings.invalid : '✗ Invalid phone number.')
          + (errTxt ? (' — ' + errTxt) : ''),
        'red'
      );
      showStatusText(row, true);
      inputEl.setAttribute('aria-invalid', 'true');
      clearWhatsAppChip(row);
      // Optional best-effort carrier guess even if invalid
      maybeLookupCarrier(true);
    }
  }

  // ==========================
  // Initialize an input field
  // ==========================
  function setupIntlTelInputOnField(inputEl) {
    if (!inputEl || inputEl.dataset.intlTelInitialized === 'true') return;

    var opts = {
      initialCountry: settings.enableIpDetection ? 'auto' : (settings.defaultCountry || 'us'),
      separateDialCode: true,
      autoPlaceholder: 'polite',
      formatOnDisplay: true,
      nationalMode: false,
      utilsScript: (settings.utilsUrl || LEGACY_UTILS_FALLBACK) // harmless for newer builds
    };

    if (Array.isArray(settings.allowedCountries) && settings.allowedCountries.length > 0) {
      opts.onlyCountries = settings.allowedCountries;
    }

    if (settings.enableIpDetection) {
      opts.geoIpLookup = function (callback) {
        fetch('https://ipapi.co/country/')
          .then(function (resp) { return resp.text(); })
          .then(function (cc) {
            cc = (cc || '').toLowerCase();
            if (Array.isArray(settings.allowedCountries) && settings.allowedCountries.length > 0) {
              if (settings.allowedCountries.indexOf(cc) === -1) {
                cc = (settings.defaultCountry || 'us');
              }
            }
            callback(cc);
          })
          .catch(function () { callback(settings.defaultCountry || 'us'); });
      };
    }

    var iti = window.intlTelInput(inputEl, opts);

    // Respect existing value or enforce default (when IP detection is OFF)
    if (inputEl.value) { try { iti.setNumber(inputEl.value); } catch (_) {} }
    else if (!settings.enableIpDetection && settings.defaultCountry) { try { iti.setCountry(settings.defaultCountry); } catch (_) {} }

    inputEl.dataset.intlTelInitialized = 'true';

    // Alignment + first status render
    applyAlignmentClass(inputEl);
    updateStatus(inputEl, iti);

    // Events
    inputEl.addEventListener('input', function () { updateStatus(inputEl, iti); });
    inputEl.addEventListener('countrychange', function () { applyAlignmentClass(inputEl); updateStatus(inputEl, iti); });

    inputEl.addEventListener('blur', function () {
      normalizeIntoField(inputEl, iti);
      loadUtilsOnce(function () { updateStatus(inputEl, iti); });
    });

    // Normalize all before submit
    var form = inputEl.form || document.querySelector('form.woocommerce-checkout') || document.querySelector('form');
    if (form && !form.dataset.wawpPhoneHooked) {
      form.addEventListener('submit', function () {
        document.querySelectorAll('input[type="tel"], input[name="billing_phone"], input#billing-phone')
          .forEach(function (el) {
            if (el && el.dataset.intlTelInitialized === 'true') {
              try {
                var inst = window.intlTelInputGlobals && window.intlTelInputGlobals.getInstance
                  ? window.intlTelInputGlobals.getInstance(el)
                  : null;
                if (inst) normalizeIntoField(el, inst);
              } catch (_) {}
            }
          });
      }, true);
      form.dataset.wawpPhoneHooked = '1';
    }

    // Improve validation once utils arrive
    loadUtilsOnce(function () { updateStatus(inputEl, iti); });
  }

  // ==========================
  // Attach to known fields
  // ==========================
  function observeIntlTelInputs() {
    var billingPhone = document.querySelector('input[name="billing_phone"]') || document.querySelector('input#billing-phone');
    if (billingPhone) setupIntlTelInputOnField(billingPhone);

    var billingPhonePopup = document.querySelector('input[name="billing_phone_popup"]') || document.querySelector('input#billing_phone_popup');
    if (billingPhonePopup) setupIntlTelInputOnField(billingPhonePopup);

    var awpNewPhone = document.querySelector('input#awp-new-phone') || document.querySelector('input[name="awp-new-phone"]');
    if (awpNewPhone) setupIntlTelInputOnField(awpNewPhone);

    var awp_user_phone = document.querySelector('input#awp_user_phone') || document.querySelector('input[name="awp_user_phone"]');
    if (awp_user_phone) setupIntlTelInputOnField(awp_user_phone);

    var whatsappEl = document.querySelector('input#awp_whatsapp');
    if (whatsappEl) setupIntlTelInputOnField(whatsappEl);

    var awpPhoneEl = document.querySelector('input#awp_phone');
    if (awpPhoneEl) setupIntlTelInputOnField(awpPhoneEl);

    var miaPhoneEl = document.querySelector('input#mia_phone');
    if (miaPhoneEl) setupIntlTelInputOnField(miaPhoneEl);

    var customPhoneEl = document.querySelector('input#awp-phone-number') || document.querySelector('input.awp-phone-number');
    if (customPhoneEl) setupIntlTelInputOnField(customPhoneEl);

    try {
      if (Array.isArray(settings.phoneFields)) {
        settings.phoneFields.forEach(function (f) {
          if (!f || f.enabled !== '1' || !f.id) return;
          var el = document.querySelector(f.id);
          if (el) setupIntlTelInputOnField(el);
        });
      }
    } catch (_) {}
  }

  function observeMutations() {
    var observer = new MutationObserver(function () { observeIntlTelInputs(); });
    observer.observe(document.body, { childList: true, subtree: true });
  }

  document.addEventListener('DOMContentLoaded', function () {
    observeIntlTelInputs();
    observeMutations();
  });
})();
